r"""
Prompt1
Newton法が使えるプログラムを配布したいです。
van der Pauw法で抵抗率を補正する経常因子f(Rab,cd/Rad,bc)をNewton法で計算し、
Rab,cd/Rad,bcに対しプロットするプログラムを作ってください

Prompt2
収束エラーです。r=1でf=1なので、そこからrを変えながら計算したほうがいいのではないでしょうか
>python vdPauw_f.py
Traceback (most recent call last):
  File "vdPauw_f.py", line 30, in <module>
    f_values = [solve_f(r) for r in r_values]
                ^^^^^^^^^^
  File "dPauw_f.py", line 25, in solve_f
    return newton(vdp_f_func, x0=0.9, args=(r,))
           ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
  File "(cut)AppData\(cut)\Python312\Lib\site-packages\scipy\optimize\_zeros_py.py", line 391, in newton
    raise RuntimeError(msg)
RuntimeError: Failed to converge after 50 iterations, value is 0.9001915296267972.
"""


import numpy as np
import matplotlib.pyplot as plt
from scipy.optimize import newton
import pandas as pd


def vdp_residual(f, r):
    """
    van der Pauwの方程式の残差:
    exp(-ln2/f * 2r/(r+1)) + exp(-ln2/f * 2/(r+1)) - 1 = 0
    """
    ln2 = np.log(2.0)
    term1 = np.exp(-(ln2 / f) * (2 * r / (r + 1)))
    term2 = np.exp(-(ln2 / f) * (2 / (r + 1)))
    return term1 + term2 - 1.0

def vdp_derivative(f, r):
    """
    Newton法の収束を高速化・安定化するための導関数 (d(residual)/df)
    """
    ln2 = np.log(2.0)
    A = ln2 * (2 * r / (r + 1))
    B = ln2 * (2 / (r + 1))
    # 指数関数の微分を活用
    df1 = np.exp(-A / f) * (A / f**2)
    df2 = np.exp(-B / f) * (B / f**2)
    return df1 + df2

def solve_f_sequence(r_list):
    """
    r=1 (f=1) を起点として、直前の解を初期値に使いながら連続的に解く
    """
    f_results = []
    current_f = 1.0  # 起点: r=1 のとき f=1
    
    for r in r_list:
        try:
            # fprime(導関数) を与えることで非常に高速かつ安定して収束します
            sol = newton(vdp_residual, x0=current_f, fprime=vdp_derivative, 
                         args=(r,), tol=1e-12, maxiter=100)
            f_results.append(sol)
            current_f = sol # 今回の解を次の初期値にする
        except RuntimeError:
            f_results.append(np.nan)
            
    return np.array(f_results)

# --- メイン処理 ---
# 1. 抵抗比 r を 1 から 1000 まで対数スケールで生成
r_values = np.logspace(0, 3, 300) # 10^0=1 から 10^3=1000 まで

# 2. 継続法によるNewton法の実行
f_values = solve_f_sequence(r_values)

df_out = pd.DataFrame({"r": r_values, "f": f_values})
df_out.to_csv("vdp_f_results.csv", index=False)
print("✅ 計算結果を 'vdp_f_results.csv' に保存しました。")

# 3. 結果のプロット
plt.figure(figsize=(8, 5))
plt.semilogx(r_values, f_values, 'b-', linewidth=2, label="Correction factor $f$")
plt.xlabel("Resistance Ratio $r = R_{ab,cd} / R_{ad,bc}$", fontsize=12)
plt.ylabel("Correction Factor $f$", fontsize=12)
plt.title("van der Pauw Correction Factor $f$ (Newton Method)", fontsize=14)
plt.grid(True, which="both", linestyle="--", alpha=0.6)
plt.axhline(1.0, color='red', linestyle=':', label="Ideal case ($r=1, f=1$)")
plt.legend()
plt.tight_layout()

# 画像とデータの保存
plt.savefig("vdp_f_plot.png")
print("✅ プロット図を 'vdp_f_plot.png' に保存しました。")
plt.show()

