import numpy as np
import math
import argparse
import matplotlib.pyplot as plt

# 物理定数
KB = 1.380649e-23
QE = 1.60217663e-19
EPS0 = 8.854187e-12
ME = 9.109383e-31

def calculate_params(args, ndn_override=None):
    T = args.temp
    ndn = ndn_override if ndn_override else args.ndn
    reff = 1.20173e6 * args.men 
    am_cm2 = reff * 1e-4        
    nc = 2.51e19 * (args.men**1.5) * ((T/300)**1.5)
    efn = args.ecn - (KB * T / QE) * math.log(ndn / nc)
    phi_b = -(args.ecn - args.efm)
    v_bi = -(efn - args.efm)
    sigma_n = QE * ndn * args.mun
    rn = 1.0 / sigma_n * args.dn * 1e-7
    js = am_cm2 * T**2 * math.exp(-QE * phi_b / (KB * T))
    return js, v_bi, phi_b, nc, rn

def calculate_schottky_current(model_type, v_target, temp, mus, ncs, nds, epss, js, phi_b, v_bi, n_diode, rs, area, v_initial, men):
    if v_target == 0: return 0.0, 0.0
    v_diode = v_initial
    dv, eps, max_iter = 1e-4, 1e-7, 100
    vt = (n_diode * KB * temp) / QE
    nds_m = nds * 1e6

    def get_current(vd):
        if vd >= 0 or model_type == 'Simple':
            return js * (math.exp(vd / vt) - 1.0) * area
        
        elif model_type == 'TFE':
            hbar = 1.0545718e-34
            m_eff = men * ME
            e00_j = (QE * hbar / 2.0) * math.sqrt(nds_m / (epss * m_eff))
            e00 = e00_j / QE
            kt_q = (KB * temp) / QE
            e0 = e00 * (1.0 / math.tanh(e00 / kt_q))
            
            # 逆方向のトンネル増幅係数 (FNトンネル的挙動のシミュレート)
            # 電界とE00に依存して障壁透過率を模した補正を入れる
            tunnel_factor = 1.0
            if vd < 0:
                tunnel_factor = math.exp(abs(vd) * (e00 / kt_q)**1.5)
            
            return js * tunnel_factor * (math.exp(vd / e0) - 1.0) * area
        
        return js * (math.exp(vd / vt) - 1.0) * area

    for _ in range(max_iter):
        im = get_current(v_diode - dv)
        ip = get_current(v_diode + dv)
        v_total = 0.5 * ((v_diode - dv + im * rs) + (v_diode + dv + ip * rs))
        if abs(v_total - v_target) < eps:
            return v_diode, 0.5 * (im + ip)
        dv_dv = ((v_diode + dv + ip * rs) - (v_diode - dv + im * rs)) / (2.0 * dv)
        v_diode += (v_target - v_total) / dv_dv
    return v_diode, get_current(v_diode)

def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('--model', choices=['Simple', 'TFE'], default='TFE')
    parser.add_argument('--temp', type=float, default=300.0)
    parser.add_argument('--area', type=float, default=0.01)
    parser.add_argument('--efm', type=float, default=4.4)
    parser.add_argument('--ecn', type=float, default=4.05)
    parser.add_argument('--mun', type=float, default=1500.0)
    parser.add_argument('--eps_r', type=float, default=11.9)
    parser.add_argument('--dn', type=float, default=1000.0)
    parser.add_argument('--men', type=float, default=0.19)
    parser.add_argument('--v0', type=float, default=-2.0)
    parser.add_argument('--v1', type=float, default=1.0)
    parser.add_argument('--step', type=float, default=0.02)
    args = parser.parse_args()

    # 比較したいドーピング濃度のリスト
    nd_list = [1e16, 1e17, 1e18, 5e18]
    epss = args.eps_r * EPS0

    plt.figure(figsize=(10, 7))

    for nd in nd_list:
        js, v_bi, phi_b, ncs, rs_unit = calculate_params(args, ndn_override=nd)
        rs = rs_unit / args.area
        v_apps = np.arange(args.v0, args.v1 + args.step, args.step)
        currents = []
        v_prev = args.v0 * 0.2
        
        for v_app in v_apps:
            v_d, i = calculate_schottky_current(
                args.model, v_app, args.temp, args.mun, ncs, nd, epss, 
                js, phi_b, v_bi, 1.0, rs, args.area, v_prev, args.men
            )
            currents.append(i)
            v_prev = v_d
        
        plt.semilogy(v_apps, np.abs(currents), label=f'Nd = {nd:.0e} cm-3')

    plt.xlabel("Applied Voltage [V]")
    plt.ylabel("|Current| [A]")
    plt.grid(True, which='both', alpha=0.3)
    plt.axvline(0, color='k', linestyle='--', alpha=0.5)
    plt.legend()
    plt.title(f"Schottky IV: Doping Concentration Dependence ({args.model} model)")
    
    print("Simulation completed. Doping levels compared:", nd_list)
    plt.show()

if __name__ == "__main__":
    main()