import numpy as np
from scipy.optimize import curve_fit
import matplotlib.pyplot as plt
import argparse


def gaussian(x, A, mu, sigma):
    return A * np.exp(-(x - mu)**2 / (2 * sigma**2))


def baseline(x, b0, b1):
    return b0 + b1 * x


def model(x, A, mu, sigma, b0, b1):
    return gaussian(x, A, mu, sigma) + baseline(x, b0, b1)


def estimate_initial_params(x, y):
    A0 = y.max() - y.min()
    mu0 = x[np.argmax(y)]
    sigma0 = (x.max() - x.min()) / 10
    b0_0 = y.min()
    b1_0 = 0.0
    return A0, mu0, sigma0, b0_0, b1_0


def peak_fit(x, y, p0):
    print("=== 初期パラメータ ===")
    print(f"A0     = {p0[0]:.3f}")
    print(f"mu0    = {p0[1]:.3f}")
    print(f"sigma0 = {p0[2]:.3f}")
    print(f"b0_0   = {p0[3]:.3f}")
    print(f"b1_0   = {p0[4]:.3f}")
    print()

    popt, pcov = curve_fit(model, x, y, p0=p0)

    # --- 標準偏差（誤差） ---
    perr = np.sqrt(np.diag(pcov))

    print("=== 最適化パラメータ（± 誤差） ===")
    print(f"A     = {popt[0]:.3f} ± {perr[0]:.3f}")
    print(f"mu    = {popt[1]:.3f} ± {perr[1]:.3f}")
    print(f"sigma = {popt[2]:.3f} ± {perr[2]:.3f}")
    print(f"b0    = {popt[3]:.3f} ± {perr[3]:.3f}")
    print(f"b1    = {popt[4]:.3f} ± {perr[4]:.3f}")
    print()

    y_init = model(x, *p0)
    y_fit = model(x, *popt)

    return y_init, y_fit


def main():
    parser = argparse.ArgumentParser(
        description="Peak fitting with optional initial parameters",
        formatter_class=argparse.ArgumentDefaultsHelpFormatter
    )

    parser.add_argument("--A0", type=float, help="Initial amplitude")
    parser.add_argument("--mu0", type=float, help="Initial peak center")
    parser.add_argument("--sigma0", type=float, help="Initial sigma")
    parser.add_argument("--b0", type=float, help="Initial baseline offset")
    parser.add_argument("--b1", type=float, help="Initial baseline slope")

    parser.add_argument("--data", type=str, required=False,
                        help="CSV file with two columns: x,y")

    args = parser.parse_args()

    # --- データ読み込み ---
    if args.data:
        data = np.loadtxt(args.data, delimiter=",")
        x, y = data[:, 0], data[:, 1]
    else:
        x = np.linspace(0, 10, 200)
        y_true = model(x, 5, 5, 0.8, 1.0, 0.1)
        noise = 0.3 * np.random.randn(len(x))
        y = y_true + noise

    # --- 初期値の決定 ---
    auto_A0, auto_mu0, auto_sigma0, auto_b0, auto_b1 = estimate_initial_params(x, y)

    A0 = args.A0 if args.A0 is not None else auto_A0
    mu0 = args.mu0 if args.mu0 is not None else auto_mu0
    sigma0 = args.sigma0 if args.sigma0 is not None else auto_sigma0
    b0 = args.b0 if args.b0 is not None else auto_b0
    b1 = args.b1 if args.b1 is not None else auto_b1

    p0 = [A0, mu0, sigma0, b0, b1]

    # --- フィット ---
    y_init, y_fit = peak_fit(x, y, p0)

    # --- プロット ---
    plt.figure(figsize=(9, 6))
    plt.plot(x, y, "o", ms=4, label="data")
    plt.plot(x, y_init, "--", lw=2, label="initial guess")
    plt.plot(x, y_fit, "-", lw=2, label="optimized fit")

    plt.legend()
    plt.xlabel("x")
    plt.ylabel("y")
    plt.title("Peak Fit: Data vs Initial Guess vs Optimized Fit")
    plt.tight_layout()
    plt.show()

    # --- 終了時に usage を表示 ---
    print("\n=== Usage ===")
    parser.print_help()


if __name__ == "__main__":
    main()
