#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
prompt
機能拡張をお願いします 
# フィッティングに使うT範囲を、Tfitmin, Tfixmaxで指定できるように。デフォルトは-1e100,+1e100にする 
# mode=fitにパラメータ誤差を計算する 
# 誤差伝播法により、各データの推定誤差を計算し、グラフでは範囲を薄水色で表示する
# 共分散行列と相関係数と固有値・固有ベクトルの出力を追加し、誤差の大きさを含めて、固定すべきパラメータの提案をする
"""


import pandas as pd
import numpy as np
import matplotlib.pyplot as plt
from scipy.optimize import minimize
import argparse
import os
import json

# 定数: ボルツマン定数 (eV/K)
K_B = 8.617333262e-5


# ---------------------------------------------------------
# 1. データ入出力関連の関数
# ---------------------------------------------------------
def load_hall_data(file_path):
    if not os.path.exists(file_path):
        print(f"エラー: ファイル '{file_path}' が見つかりません。")
        return None
    if file_path.endswith('.csv'):
        return pd.read_csv(file_path)
    return pd.read_excel(file_path)


def save_params(params, filename='fit_params.json'):
    with open(filename, 'w', encoding='utf-8') as f:
        json.dump(params, f, indent=4, ensure_ascii=False)
    print(f"パラメータを保存しました: {filename}")


def load_params(filename='fit_params.json'):
    if not os.path.exists(filename):
        alt_filename = 'llsq_params.json'
        if os.path.exists(alt_filename):
            filename = alt_filename
        else:
            print(f"警告: パラメータファイルが見つかりません。デフォルト値を使用します。")
            return {'aop': 1e-3, 'a1': 1e-7, 'a2': 1e-4, 'a3': 1e1, 'VB': 0.0}
    with open(filename, 'r', encoding='utf-8') as f:
        return json.load(f)


# ---------------------------------------------------------
# 2. 物理モデルと解析関連の関数
# ---------------------------------------------------------
def get_inv_mu_components(T, params, Eop):
    """
    params: [aop, a1, a2, a3, VB]
    """
    aop, a1, a2, a3, VB = params

    # Optical phonon factor (Bose-Einstein)
    f_op = 1.0 / (np.exp(Eop / (K_B * T)) - 1.0)
    f_ac = T**1.5
    f_ni = np.ones_like(T)
    f_ii = T**-1.5

    components = {
        'Optical Phonon': aop * f_op,
        'Acoustic Phonon': a1 * f_ac,
        'Neutral Impurity': a2 * f_ni,
        'Ionized Impurity': a3 * f_ii
    }

    inv_mu_bulk = np.maximum(sum(components.values()), 1e-30)

    # exp(VB/kT) can overflow for small T; clip exponent safely.
    expo = VB / (K_B * T)
    expo = np.clip(expo, -700.0, 700.0)
    exp_factor = np.exp(expo)

    inv_mu_total = inv_mu_bulk * exp_factor
    components['Grain Boundary'] = inv_mu_total - inv_mu_bulk

    return components, inv_mu_total


def solve_llsq(T, mu_exp, Eop):
    f_op = 1.0 / (np.exp(Eop / (K_B * T)) - 1.0)
    f_ac = T**1.5
    f_ni = np.ones_like(T)
    f_ii = T**-1.5
    X = np.column_stack([f_op, f_ac, f_ni, f_ii])
    y = 1.0 / mu_exp
    coeffs, _, _, _ = np.linalg.lstsq(X, y, rcond=None)
    # 負の値にならないようクリップ
    return np.maximum(coeffs, 0)


def fit_mask(T, Tfitmin=-1e100, Tfitmax=+1e100):
    return (T >= Tfitmin) & (T <= Tfitmax)


def build_full_params(init_full, optimize_indices, p_opt):
    p = list(init_full)
    for idx, val in zip(optimize_indices, p_opt):
        p[idx] = val
    return p


def residuals_log10(T, mu_exp, params_full, Eop):
    """r_i = log10(mu_exp) - log10(mu_model)."""
    _, inv_total = get_inv_mu_components(T, params_full, Eop)
    mu_model = 1.0 / np.maximum(inv_total, 1e-300)
    # Avoid log of non-positive (should not happen, but guard)
    mu_model = np.maximum(mu_model, 1e-300)
    mu_exp_safe = np.maximum(mu_exp, 1e-300)
    return np.log10(mu_exp_safe) - np.log10(mu_model)


def numerical_jacobian(fun_vec, p, rel_step=1e-6, abs_step=1e-12):
    """
    Central-difference Jacobian for vector-valued function.
    fun_vec(p) -> (N,) residual vector
    Returns J shape (N, M)
    """
    p = np.asarray(p, dtype=float)
    f0 = np.asarray(fun_vec(p), dtype=float)
    n = f0.size
    m = p.size
    J = np.zeros((n, m), dtype=float)

    for j in range(m):
        dp = rel_step * (abs(p[j]) + 1.0) + abs_step
        p1 = p.copy(); p1[j] += dp
        p2 = p.copy(); p2[j] -= dp
        f1 = np.asarray(fun_vec(p1), dtype=float)
        f2 = np.asarray(fun_vec(p2), dtype=float)
        J[:, j] = (f1 - f2) / (2.0 * dp)
    return J


def param_covariance(residual_vec, J, dof):
    """
    Covariance of parameters (approx.): cov = s^2 (J^T J)^{-1}
    residual_vec: (N,)
    J: (N, M) Jacobian of residuals
    dof: N - M
    """
    if dof <= 0:
        return None, None
    s2 = float(np.sum(residual_vec**2) / dof)
    JTJ = J.T @ J
    try:
        cov = s2 * np.linalg.inv(JTJ)
    except np.linalg.LinAlgError:
        cov = s2 * np.linalg.pinv(JTJ)
    stderr = np.sqrt(np.maximum(np.diag(cov), 0.0))
    return cov, stderr



# ============================================================
# covariance / correlation / eigen diagnostics
# ============================================================
def cov_to_corr(cov):
    """Convert covariance matrix -> (corr, std)."""
    cov = np.asarray(cov, dtype=float)
    std = np.sqrt(np.maximum(np.diag(cov), 0.0))
    denom = np.outer(std, std)
    with np.errstate(invalid='ignore', divide='ignore'):
        corr = cov / denom
    np.fill_diagonal(corr, 1.0)
    # where std==0 -> corr undefined; set to nan except diagonal
    z = denom == 0
    corr[z] = np.nan
    np.fill_diagonal(corr, 1.0)
    return corr, std

def eigen_sorted_sym(A):
    """Eigen-decomposition of symmetric matrix A; returns (evals_desc, evecs_desc)."""
    A = np.asarray(A, dtype=float)
    evals, evecs = np.linalg.eigh(A)
    idx = np.argsort(evals)[::-1]  # descending
    return evals[idx], evecs[:, idx]

def summarize_eigenvectors(evals, evecs, names, topk=3, compk=3):
    """
    Create a human-readable summary list for the top eigenvectors.
    Each item: {'rank', 'eigenvalue', 'components': [(name, weight), ...]}
    """
    names = list(names)
    out = []
    k = min(topk, len(evals))
    for r in range(k):
        v = evecs[:, r]
        order = np.argsort(np.abs(v))[::-1]
        comps = []
        for j in order[:min(compk, len(order))]:
            comps.append((names[j], float(v[j])))
        out.append({
            'rank': r + 1,
            'eigenvalue': float(evals[r]),
            'components': comps
        })
    return out

def propose_fix_candidates(names, values, stderr, corr, evals_cov=None, evecs_cov=None,
                           corr_thr=0.95, relerr_thr=0.5, topn=3):
    """
    Heuristic proposal of parameters to fix / constrain.
    - Considers large relative errors
    - Strong pairwise correlations
    - Dominant components in the most-uncertain eigen-direction of covariance
    Returns list of dict: {'param', 'score', 'reasons':[...]}
    """
    names = list(names)
    values = np.asarray(values, dtype=float)
    stderr = np.asarray(stderr, dtype=float)

    tiny = 1e-30
    relerr = stderr / (np.abs(values) + tiny)

    # base score from relative error
    score = relerr.copy()
    reasons = {n: [] for n in names}

    for i, n in enumerate(names):
        if not np.isfinite(relerr[i]):
            continue
        if relerr[i] >= relerr_thr:
            reasons[n].append(f"relative stderr = {relerr[i]:.3g} (>= {relerr_thr})")
        else:
            reasons[n].append(f"relative stderr = {relerr[i]:.3g}")

    # correlation contribution
    corr = np.asarray(corr, dtype=float)
    for i in range(len(names)):
        for j in range(i+1, len(names)):
            cij = corr[i, j]
            if not np.isfinite(cij):
                continue
            if abs(cij) >= corr_thr:
                # add score to both; slightly prefer the worse-determined one later by larger score
                bonus = (abs(cij) - corr_thr) * 5.0 + 0.5
                score[i] += bonus
                score[j] += bonus
                reasons[names[i]].append(f"strong corr with {names[j]}: {cij:+.3f} (>= {corr_thr})")
                reasons[names[j]].append(f"strong corr with {names[i]}: {cij:+.3f} (>= {corr_thr})")

    # eigen-direction (largest variance direction)
    if evals_cov is not None and evecs_cov is not None and len(evals_cov) > 0:
        v = evecs_cov[:, 0]  # eigenvector for largest eigenvalue (most uncertain direction)
        w = np.abs(v)
        w = w / (w.max() + tiny)
        for i, n in enumerate(names):
            if w[i] >= 0.5:
                add = 0.7 * w[i]
                score[i] += add
                reasons[n].append(f"dominant in most-uncertain eigen-direction: |v|/max={w[i]:.2f}")

    # build ranked list
    items = []
    for i, n in enumerate(names):
        items.append({
            'param': n,
            'score': float(score[i]),
            'value': float(values[i]),
            'stderr': float(stderr[i]),
            'relerr': float(relerr[i]) if np.isfinite(relerr[i]) else None,
            'reasons': reasons[n],
        })
    items.sort(key=lambda d: d['score'], reverse=True)

    # Suggest only if there is any "problem" signal
    # (large relerr or strong corr)
    filtered = []
    for it in items:
        if (it['relerr'] is not None and it['relerr'] >= relerr_thr) or any('strong corr' in r for r in it['reasons']):
            filtered.append(it)
    if not filtered:
        filtered = items[:min(topn, len(items))]
    return filtered[:min(topn, len(filtered))]


def prediction_band_log10_mu(T, params_full, Eop, optimize_indices, cov_opt, nsigma=1.0,
                             rel_step=1e-6, abs_step=1e-12):
    """
    Error propagation (delta method) for y(T)=log10(mu_model(T)).
    Returns (yhat, ylow, yhigh) arrays of shape (len(T),)
    Only optimized parameters contribute via cov_opt.
    """
    _, inv0 = get_inv_mu_components(T, params_full, Eop)
    mu0 = 1.0 / np.maximum(inv0, 1e-300)
    y0 = np.log10(np.maximum(mu0, 1e-300))

    m = len(optimize_indices)
    if cov_opt is None or m == 0:
        return y0, None, None

    # Gradients dy/dp for optimized params
    grads = np.zeros((len(T), m), dtype=float)

    p0_opt = np.array([params_full[i] for i in optimize_indices], dtype=float)

    for j in range(m):
        dp = rel_step * (abs(p0_opt[j]) + 1.0) + abs_step

        # +dp
        p_plus = p0_opt.copy(); p_plus[j] += dp
        params_plus = list(params_full)
        for idx, val in zip(optimize_indices, p_plus):
            params_plus[idx] = val
        _, invp = get_inv_mu_components(T, params_plus, Eop)
        mup = 1.0 / np.maximum(invp, 1e-300)
        yp = np.log10(np.maximum(mup, 1e-300))

        # -dp
        p_minus = p0_opt.copy(); p_minus[j] -= dp
        params_minus = list(params_full)
        for idx, val in zip(optimize_indices, p_minus):
            params_minus[idx] = val
        _, invm = get_inv_mu_components(T, params_minus, Eop)
        mum = 1.0 / np.maximum(invm, 1e-300)
        ym = np.log10(np.maximum(mum, 1e-300))

        grads[:, j] = (yp - ym) / (2.0 * dp)

    # var(y) = g^T cov g
    var_y = np.einsum('ni,ij,nj->n', grads, cov_opt, grads)
    sig_y = np.sqrt(np.maximum(var_y, 0.0))

    ylow = y0 - nsigma * sig_y
    yhigh = y0 + nsigma * sig_y
    return y0, ylow, yhigh


# ---------------------------------------------------------
# 3. 可視化関連の関数
# ---------------------------------------------------------
def visualize_fit(T, mu_exp, mu_fit=None, mu_lo=None, mu_hi=None,
                  title='Hall Mobility Fit', save_name='plot.png'):
    plt.figure(figsize=(8, 6))
    plt.scatter(T, mu_exp, color='red', label='Experimental', alpha=0.6)

    idx = np.argsort(T)
    T_sorted = T[idx]

    if mu_fit is not None:
        mu_fit_sorted = np.asarray(mu_fit)[idx]
        plt.plot(T_sorted, mu_fit_sorted, color='blue', label='Model Fit', linewidth=2)

        if mu_lo is not None and mu_hi is not None:
            mu_lo_sorted = np.asarray(mu_lo)[idx]
            mu_hi_sorted = np.asarray(mu_hi)[idx]
            # 薄水色の誤差帯
            plt.fill_between(T_sorted, mu_lo_sorted, mu_hi_sorted, alpha=0.25, label='Model ±1σ')

    plt.xlabel('Temperature (K)')
    plt.ylabel('Mobility (cm²/Vs)')
#    plt.yscale('log')
    plt.title(title)
    plt.legend()
    plt.grid(True, which='both', alpha=0.3)
    plt.tight_layout()
    plt.savefig(save_name)
    plt.show()


def visualize_weights(T, components, total, save_name='weight_plot.png'):
    plt.figure(figsize=(10, 6))
    idx = np.argsort(T)
    T_sorted = T[idx]
    for name, inv_mu in components.items():
        weight = (inv_mu / total) * 100
        plt.plot(T_sorted, weight[idx], marker='o', markersize=4, label=name, linestyle='-', alpha=0.8)
    plt.xlabel('Temperature (K)')
    plt.ylabel('Contribution to Scattering (%)')
    plt.title('Scattering Mechanism Contributions')
    plt.ylim(-5, 105)
    plt.grid(True, linestyle='--', alpha=0.5)
    plt.legend(loc='best')
    plt.tight_layout()
    plt.savefig(save_name)
    plt.show()


# ---------------------------------------------------------
# 4. メイン処理
# ---------------------------------------------------------
def main():
    parser = argparse.ArgumentParser(description='Hall効果移動度フィッティング（パラメータ固定＆誤差推定付き）')
    parser.add_argument('--input', type=str, default='Hall-T1.xlsx', help='入力ファイル名')
    parser.add_argument('--temp_col', type=int, default=0, help='温度列(0開始)')
    parser.add_argument('--mu_col', type=int, default=2, help='移動度列(0開始)')
    parser.add_argument('--mode', type=str, choices=['read', 'llsq', 'fit', 'weight'], default='read')
    parser.add_argument('--method', type=str, default='Nelder-Mead')
    parser.add_argument('--eop', type=float, default=0.045, help='光学フォノンエネルギー(eV)')
    parser.add_argument('--fix', type=str, default='', help='固定するパラメータをカンマ区切りで指定 (例: a2,VB)')

    # fitting temperature window
    parser.add_argument('--Tfitmin', type=float, default=-1e100, help='フィットに使う最小温度[K] (default -1e100)')
    parser.add_argument('--Tfitmax', type=float, default=+1e100, help='フィットに使う最大温度[K] (default +1e100)')

    # error band / uncertainty controls
    parser.add_argument('--band_sigma', type=float, default=1.0, help='誤差帯の幅 (nsigma). default 1.0 (±1σ)')
    parser.add_argument('--jac_relstep', type=float, default=1e-6, help='数値微分の相対ステップ')
    parser.add_argument('--jac_absstep', type=float, default=1e-12, help='数値微分の絶対ステップ')
    args = parser.parse_args()

    print()
    print(f"input file={args.input}")
    print(f"mode={args.mode}")
    print(f"method={args.method}")
    print(f"eop={args.eop} eV")
    print(f"fix={args.fix}")
    print(f"Tfitmin={args.Tfitmin}, Tfitmax={args.Tfitmax}")

    df = load_hall_data(args.input)
    if df is None:
        return

    T = df.iloc[:, args.temp_col].values.astype(float)
    mu_exp = df.iloc[:, args.mu_col].values.astype(float)

    print()
    print("T  mu")
    for _T, _mu in zip(T, mu_exp):
        print(f"{_T:.3f}  {_mu:.4g}")

    labels = ['aop', 'a1', 'a2', 'a3', 'VB']
    fixed_list = [s.strip() for s in args.fix.split(',') if s.strip()] if args.fix else []

    if args.mode == 'read':
        print("\n--- 読み込みデータ ---")
        print(df)
        visualize_fit(T, mu_exp, title='Experimental Data')

    elif args.mode == 'llsq':
        c = solve_llsq(T, mu_exp, args.eop)
        params_dict = {'aop': float(c[0]), 'a1': float(c[1]), 'a2': float(c[2]), 'a3': float(c[3]), 'VB': 0.0}
        save_params(params_dict, filename='llsq_params.json')
        _, inv_fit = get_inv_mu_components(T, list(c) + [0.0], args.eop)
        visualize_fit(T, mu_exp, 1 / inv_fit, title='LLSQ Initial Fit (VB=0)')

    elif args.mode == 'fit':
        # --- prepare data mask ---
        mfit = fit_mask(T, args.Tfitmin, args.Tfitmax)
        if np.sum(mfit) < 3:
            print("エラー: フィットに使える点が少なすぎます。Tfitmin/Tfitmaxを見直してください。")
            return

        T_fit = T[mfit]
        mu_fit_exp = mu_exp[mfit]

        print()
        print(f"--- フィットに使う点数: {np.sum(mfit)}/{len(T)} ---")
        if not (args.Tfitmin <= -1e50 and args.Tfitmax >= 1e50):
            print(f"  使用温度範囲: {args.Tfitmin} 〜 {args.Tfitmax} K")
        print("T  mu")
        for _T, _mu in zip(T_fit, mu_fit_exp):
            print(f"{_T:.3f}  {_mu:.6g}")


        p_base = load_params()
        init_full = [float(p_base.get(l, 0.0)) for l in labels]

        optimize_indices = [i for i, l in enumerate(labels) if l not in fixed_list]
        print(f"最適化対象: {[labels[i] for i in optimize_indices]}")
        print(f"固定パラメータ: {fixed_list}")
        print("--- 初期パラメータ ---")
        for _label, _val in zip(labels, init_full):
            state = "(FIXED)" if _label in fixed_list else ""
            print(f"{_label:4s}: {_val:.6g} {state}")

        def objective(p_opt):
            p_current = build_full_params(init_full, optimize_indices, p_opt)

            # ペナルティ: aop,a1,a2,a3 は非負を期待（VBは許容）
            if any(np.array(p_current)[:-1] < 0):
                return 1e18

            r = residuals_log10(T_fit, mu_fit_exp, p_current, args.eop)
            return float(np.sum(r**2))

        init_opt = [init_full[i] for i in optimize_indices]
        res = minimize(objective, init_opt, method=args.method, options={'maxiter': 5000})

        # 結果の統合（full）
        final_values = build_full_params(init_full, optimize_indices, res.x)
        final_params = {l: float(v) for l, v in zip(labels, final_values)}
        save_params(final_params, filename='fit_params.json')

        print("\n--- 最適化結果 ---")
        print(f"success={res.success}, message={res.message}")
        print(f"SSE={res.fun:.6g}")
        for k, v in final_params.items():
            state = "(FIXED)" if k in fixed_list else ""
            print(f"{k:4s}: {v:.6g} {state}")

        # --- parameter uncertainty via Jacobian of residuals ---
        m = len(optimize_indices)
        dof = int(np.sum(mfit) - m)
        cov_opt = None
        stderr_opt = None

        if m > 0 and dof > 0:
            def r_of_popt(p_opt):
                p_full = build_full_params(init_full, optimize_indices, p_opt)
                return residuals_log10(T_fit, mu_fit_exp, p_full, args.eop)

            rvec = r_of_popt(res.x)
            J = numerical_jacobian(
                r_of_popt, np.asarray(res.x, dtype=float),
                rel_step=args.jac_relstep, abs_step=args.jac_absstep
            )
            cov_opt, stderr_opt = param_covariance(rvec, J, dof)

        # --- print and save parameter errors (expand to full labels) ---
        stderr_full = {lab: None for lab in labels}
        if stderr_opt is not None:
            for lab, idx_opt, se in zip([labels[i] for i in optimize_indices], optimize_indices, stderr_opt):
                stderr_full[labels[idx_opt]] = float(se)
        for lab in fixed_list:
            if lab in stderr_full:
                stderr_full[lab] = 0.0

        print("\n--- パラメータ誤差（近似） ---")
        print("  ※ r_i=log10(mu_exp)-log10(mu_model) を最小化したときの線形化近似（J^T J）")
        print(f"  dof = Nfit({np.sum(mfit)}) - M({m}) = {dof}")
        for lab in labels:
            v = final_params[lab]
            se = stderr_full[lab]
            if se is None:
                tag = "(NOERR)" if lab not in fixed_list else "(FIXED)"
                print(f"{lab:4s}: {v:.6g} {tag}")
            else:
                tag = "(FIXED)" if lab in fixed_list else ""
                print(f"{lab:4s}: {v:.6g}  ± {se:.3g} {tag}")

        save_params({'value': final_params, 'stderr': stderr_full}, filename='fit_params_with_errors.json')

        # ==========================================================
        # Diagnostics: covariance / correlation / eigenvalues
        # ==========================================================
        if cov_opt is not None and stderr_opt is not None and len(optimize_indices) > 0:
            opt_names = [labels[i] for i in optimize_indices]
            opt_vals = [final_values[i] for i in optimize_indices]

            # correlation matrix for optimized parameters
            corr_opt, std_opt = cov_to_corr(cov_opt)

            # eigen of covariance (principal uncertainty directions)
            evals_cov, evecs_cov = eigen_sorted_sym(cov_opt)

            # information matrix JTJ and its eigen (conditioning)
            # JTJ ≈ (s^2) * inv(cov) ; but we have J directly if we computed it
            # here we recompute JTJ from J if available in local scope
            JTJ = None
            evals_JTJ = None
            evecs_JTJ = None
            cond_JTJ = None
            try:
                # J exists only when uncertainty estimation ran
                JTJ = J.T @ J
                # for JTJ, small eigenvalues are the problematic ones -> sort ascending for reporting
                w, V = np.linalg.eigh(JTJ)
                idx_asc = np.argsort(w)
                evals_JTJ = w[idx_asc]
                evecs_JTJ = V[:, idx_asc]
                if np.all(evals_JTJ > 0):
                    cond_JTJ = float(evals_JTJ[-1] / evals_JTJ[0])
            except Exception:
                JTJ = None

            # --- print matrices (optimized params only) ---
            np.set_printoptions(precision=4, suppress=True, linewidth=140)

            print("\n=== 共分散行列（opt params）: Cov ≈ s^2 (J^T J)^{-1} ===")
            print("opt params =", opt_names)
            print(cov_opt)

            print("\n=== 相関係数行列（opt params） ===")
            print("opt params =", opt_names)
            print(corr_opt)

            if evals_cov is not None:
                print("\n=== Cov の固有値（大→小）と主要成分（上位3） ===")
                ev_summary = summarize_eigenvectors(evals_cov, evecs_cov, opt_names, topk=min(3, len(opt_names)), compk=min(4, len(opt_names)))
                for item in ev_summary:
                    comps = ", ".join([f"{n}:{w:+.3f}" for n, w in item['components']])
                    print(f"  #{item['rank']}: eigenvalue={item['eigenvalue']:.6g}  components=({comps})")

            if evals_JTJ is not None:
                print("\n=== J^T J の固有値（小→大） ===")
                print(evals_JTJ)
                if cond_JTJ is not None:
                    print(f"cond(J^T J) ≈ {cond_JTJ:.3e} (大きいほど悪条件)")

                # show the *worst* direction (smallest eigenvalue) composition
                v_bad = evecs_JTJ[:, 0]
                order = np.argsort(np.abs(v_bad))[::-1]
                comps = ", ".join([f"{opt_names[i]}:{v_bad[i]:+.3f}" for i in order[:min(4, len(order))]])
                print(f"最小固有値方向（最も決まりにくい結合）: ({comps})")

            # --- save diagnostics as json (optimized-only) ---
            diag = {
                'opt_params': opt_names,
                'cov': cov_opt.tolist(),
                'corr': corr_opt.tolist(),
                'stderr_opt': [float(x) for x in stderr_opt],
                'eig_cov': {
                    'eigenvalues_desc': [float(x) for x in evals_cov],
                    'eigenvectors_desc': evecs_cov.tolist(),
                    'summary': summarize_eigenvectors(evals_cov, evecs_cov, opt_names, topk=min(5, len(opt_names)), compk=min(5, len(opt_names))),
                }
            }
            if evals_JTJ is not None:
                diag['eig_JTJ'] = {
                    'eigenvalues_asc': [float(x) for x in evals_JTJ],
                    'eigenvectors_asc': evecs_JTJ.tolist(),
                    'cond': cond_JTJ,
                    'worst_direction': [{'param': opt_names[i], 'weight': float(evecs_JTJ[i,0])} for i in range(len(opt_names))]
                }

            save_params(diag, filename='fit_diagnostics_opt.json')

            # --- propose candidates to fix / constrain (heuristic) ---
            suggestions = propose_fix_candidates(
                opt_names, opt_vals, stderr_opt, corr_opt,
                evals_cov=evals_cov, evecs_cov=evecs_cov,
                corr_thr=0.95, relerr_thr=0.5, topn=3
            )
            save_params({'suggestions': suggestions}, filename='fit_fix_suggestions.json')

            print("\n=== 固定（または外部拘束）候補の提案（ヒューリスティック） ===")
            print("※ 数値的に『決まりにくい』指標（大きい誤差/強相関/不確か方向への寄与）からの提案です。物理的妥当性・独立測定の有無で最終判断してください。")
            for k, s in enumerate(suggestions, 1):
                re_str = f"{s['relerr']:.3g}" if s.get('relerr') is not None else "NA"
                print(f"  [{k}] {s['param']}: value={s['value']:.6g}, stderr={s['stderr']:.3g}, relerr={re_str}, score={s['score']:.3g}")
                for r in s.get('reasons', [])[:6]:
                    print(f"       - {r}")

        else:
            print("\n(診断) 共分散/相関の推定に必要な条件を満たさないため、診断出力をスキップしました。")
            print("       例: フィット点数が少ない / 全パラメータ固定 / dof<=0 など")



        # --- model curve and propagated uncertainty band on ALL T (for plotting) ---
        _, inv_all = get_inv_mu_components(T, final_values, args.eop)
        mu_model_all = 1.0 / inv_all

        y0, ylo, yhi = prediction_band_log10_mu(
            T, final_values, args.eop, optimize_indices, cov_opt,
            nsigma=float(args.band_sigma),
            rel_step=args.jac_relstep, abs_step=args.jac_absstep
        )

        mu_lo = None
        mu_hi = None
        if ylo is not None and yhi is not None:
            mu_lo = 10.0 ** ylo
            mu_hi = 10.0 ** yhi

        visualize_fit(
            T, mu_exp,
            mu_fit=mu_model_all,
            mu_lo=mu_lo,
            mu_hi=mu_hi,
            title=f'Final Fit (band=±{args.band_sigma}σ, fit-range=[{args.Tfitmin},{args.Tfitmax}] K)',
            save_name='mu_vs_T_fit.png'
        )

        # --- estimated per-point uncertainty table (only for original data points) ---
        if mu_lo is not None and mu_hi is not None:
            print("\n--- 各データ点の推定誤差（誤差伝播, ±1σ を log10(mu) で評価） ---")
            print("# T[K]      mu_model      mu_lo        mu_hi")
            for _T, _m, _lo, _hi in zip(T, mu_model_all, mu_lo, mu_hi):
                print(f"{_T:8.3f}  {_m:11.4g}  {_lo:11.4g}  {_hi:11.4g}")

    elif args.mode == 'weight':
        p_dict = load_params()
        p_list = [float(p_dict.get(l, 0.0)) for l in labels]
        comp, total = get_inv_mu_components(T, p_list, args.eop)
        visualize_weights(T, comp, total, save_name='mu_weight_plot.png')


if __name__ == '__main__':
    main()


