import sys
import numpy as np
from scipy.integrate import simpson, quad
from numpy.polynomial.legendre import leggauss
import time


# 積分範囲
a, b = 0.0, 5.0
# 分割数（点数）
n = 10  

argv = sys.argv
nargs = len(argv)
if nargs > 1: n = int(argv[1])
if nargs > 2: a = float(argv[2])
if nargs > 3: b = float(argv[3])


# 積分する関数
def func(x):
    return np.exp(-x*x)


# Riemann 和（左リーマン）
def riemann_sum(f, a, b, n):
    x = np.linspace(a, b, n)
    dx = (b - a) / (n - 1)
    return np.sum(f(x)) * dx


# 台形則
def trapezoid_rule(f, a, b, n):
    x = np.linspace(a, b, n)
    y = f(x)
    return np.trapezoid(y, x)


# Simpson 則
def simpson_rule(f, a, b, n):
    x = np.linspace(a, b, n)
    y = f(x)
    return simpson(y, x=x)


# Gauss–Legendre 法
def gauss_legendre_integrate(f, a, b, n):
    xg, wg = leggauss(n)  # [-1,1] の節点と重み
    x = 0.5 * (b - a) * xg + 0.5 * (b + a)
    w = 0.5 * (b - a) * wg
    return np.sum(f(x) * w)


# ベンチマーク用デコレータ
def benchmark(func, *args):
    start = time.perf_counter()
    result = func(*args)
    elapsed = (time.perf_counter() - start) * 1000  # ms
    return result, elapsed


# -----------------------------
# メイン処理
# -----------------------------
true_value = 0.886226925

print(f"Integrating f(x)=exp(-x*x) from {a} to {b} with divide number = {n}\n")
print(f"True value: {true_value:.10f}\n")

methods = [
    ("Riemann sum", riemann_sum, (func, a, b, n)),
    ("Trapezoid rule", trapezoid_rule, (func, a, b, n)),
    ("Simpson rule", simpson_rule, (func, a, b, n)),
    ("Gauss-Legendre", gauss_legendre_integrate, (func, a, b, n)),
    ("quad", lambda f, a, b, n: quad(f, a, b)[0], (func, a, b, n)),
]

for name, method, args in methods:
    value, t = benchmark(method, *args)
    error = abs(value - true_value)
    print(f"{name:15s} value={value:.10f}  error={error:.2e}  time={t:.3f} ms")
