import numpy as np
import matplotlib.pyplot as plt
from scipy.integrate import quad
from scipy.special import expit, gamma
from functools import lru_cache

# =========================
# constants (SI)
# =========================
kB = 1.380649e-23          # J/K
e  = 1.602176634e-19       # C
h  = 6.62607015e-34        # J s
m0 = 9.1093837015e-31      # kg
kB_eV = 8.617333262145e-5  # eV/K

# ============================================================
# Fermi–Dirac integral (your convention: NO 1/Gamma(j+1))
#   Fj(j,xi) = ∫_0^∞ x^j / (1+exp(x-xi)) dx
# ============================================================
@lru_cache(maxsize=20000)
def Fj(j, xi):
    j = float(j)
    xi = float(np.round(xi, 12))
    integrand = lambda x: (x**j) * expit(xi - x)
    val, _ = quad(integrand, 0.0, np.inf, epsabs=1e-10, epsrel=1e-10, limit=400)
    return val

def Nc_3D(T, m_eff=1.0):
    """
    Effective density of states Nc [m^-3] for 3D parabolic band:
      Nc = 2 * (2π m* kB T / h^2)^(3/2)
    """
    mstar = m_eff * m0
    return 2.0 * (2.0 * np.pi * mstar * kB * T / (h**2))**1.5

def electron_density_from_xi(xi, T, m_eff=1.0):
    """
    n [m^-3] = Nc * F_{1/2}^{std}(xi)
    with F^{std}_j(xi) = (1/Gamma(j+1)) * ∫ x^j/(1+exp(x-xi)) dx
    Our Fj is the integral only, so:
      F^{std}_{1/2} = Fj(1/2,xi) / Gamma(3/2)
    """
    Nc = Nc_3D(T, m_eff=m_eff)
    F12_std = Fj(0.5, xi) / gamma(1.5)
    return Nc * F12_std

def seebeck_from_xi(xi, r, T, carrier="electron"):
    """
    S [V/K] with your formula:
      S = (kB/q) * [ (r+2)/(r+1) * Fj(r+1,xi)/Fj(r,xi) - xi ]
    where q = -e for electrons, +e for holes.
    """
    if carrier.lower().startswith("e"):
        q = -e
    elif carrier.lower().startswith("h"):
        q = +e
    else:
        raise ValueError("carrier must be 'electron' or 'hole'")

    Fr  = Fj(r, xi)
    Fr1 = Fj(r + 1.0, xi)
    bracket = ((r + 2.0) / (r + 1.0)) * (Fr1 / Fr) - xi
    return (kB / q) * bracket

def main():
    # -------------------------
    # user settings
    # -------------------------
    T = 300.0          # K
    m_eff = 1.0        # effective mass m*/m0 (change as needed)
    carrier = "electron"

    # r values in YOUR Seebeck formula (same r as your slide)
    r_list = [2.0, 1.5, 1.0, 0.5, 0.0, -0.5]

    # scan xi = (EF-Ec)/kBT (dimensionless)
    # wide range covers nondegenerate -> degenerate
    xi_range = np.linspace(-5.0, 60.0, 241)

    # -------------------------
    # compute & plot
    # -------------------------
    plt.figure(figsize=(8, 6))

    for r in r_list:
        n_list = []
        S_list = []
        for xi in xi_range:
            n = electron_density_from_xi(xi, T=T, m_eff=m_eff)   # [m^-3]
            S = seebeck_from_xi(xi, r=r, T=T, carrier=carrier)   # [V/K]
            n_list.append(n)
            S_list.append(S)

        n_arr = np.array(n_list) / 1e6       # -> [cm^-3]
        S_arr = np.array(S_list) * 1e6       # -> [µV/K]

        # n increases monotonically with xi, so plotting is safe
        plt.plot(n_arr, S_arr, label=f"r = {r:g}")

    plt.xscale("log")
    plt.axhline(0.0, linewidth=1.0, alpha=0.4)
    plt.xlabel(r"$N_e$ (cm$^{-3}$)")
    plt.ylabel(r"$S$ ($\mu$V/K)")
    plt.title(f"S vs Ne (T={T:g} K, m*/m0={m_eff:g}, {carrier})")
    plt.grid(True, which="both", alpha=0.25)
    plt.legend()
    plt.tight_layout()
    plt.show()

if __name__ == "__main__":
    main()
