import numpy as np
import matplotlib.pyplot as plt
from scipy.integrate import quad
from scipy.special import expit
from functools import lru_cache

# =========================
# constants
# =========================
kB_SI = 1.380649e-23           # J/K
e_SI  = 1.602176634e-19        # C
kB_eV = 8.617333262145e-5      # eV/K

# =========================
# Fermi–Dirac integral
# Fj(j,xi) = ∫_0^∞ x^j / (1+exp(x-xi)) dx
# (NO 1/Gamma(j+1) prefactor)
# =========================
@lru_cache(maxsize=20000)
def Fj(j, xi):
    # quad のキャッシュキーを安定にするため丸め
    j = float(j)
    xi = float(np.round(xi, 12))

    integrand = lambda x: (x**j) * expit(xi - x)
    val, _ = quad(integrand, 0.0, np.inf, epsabs=1e-10, epsrel=1e-10, limit=300)
    return val

def seebeck_S(EF_eV, r, T=300.0, carrier="electron"):
    """
    Seebeck coefficient S [V/K] for the definition:
      S = (kB/q) [ (r+2)/(r+1) * F_{r+1}(xi)/F_r(xi) - xi ]
    with xi = (EF - Ec)/(kBT). Here EF_eV is measured from Ec (Ec=0).
    """
    # charge sign
    if carrier.lower().startswith("e"):     # electron
        q = -e_SI
    elif carrier.lower().startswith("h"):   # hole
        q = +e_SI
    else:
        raise ValueError("carrier must be 'electron' or 'hole'")

    xi = EF_eV / (kB_eV * T)

    Fr  = Fj(r, xi)
    Fr1 = Fj(r + 1.0, xi)

    bracket = ((r + 2.0) / (r + 1.0)) * (Fr1 / Fr) - xi
    S = (kB_SI / q) * bracket   # [V/K]
    return S

def main():
    # ---- settings ----
    T = 300.0
    carrier = "electron"  # "electron" or "hole"

    EF_range = np.linspace(-0.2, 0.5, 121)  # eV, measured from Ec
    r_list = [2.0, 1.5, 1.0, 0.5, 0.0]       # your r in the formula

    # ---- plot ----
    plt.figure(figsize=(8, 6))
    for r in r_list:
        S_vals = np.array([seebeck_S(EF, r, T=T, carrier=carrier) for EF in EF_range])
        plt.plot(EF_range, S_vals * 1e6, label=f"r = {r:g}")  # microV/K

    plt.axhline(0.0, linewidth=1.0, alpha=0.4)
    plt.xlabel(r"$E_F - E_c$ (eV)")
    plt.ylabel(r"$S$ ($\mu$V/K)")
    plt.title(f"Seebeck coefficient vs Fermi level  (T={T:g} K, {carrier})")
    plt.grid(True, alpha=0.25)
    plt.legend()
    plt.tight_layout()
    plt.show()

if __name__ == "__main__":
    main()
