import numpy as np
from scipy.special import expit
from tkfermi_integral import fermi_dirac_e_high_precision, fermi_dirac_e_if, fermi_dirac_e


# 物理定数
kB = 1.380649e-23      # J/K
q  = 1.602176634e-19   # C

# -----------------------------
# NDp の2つの実装
# -----------------------------

def NDp_expit(Ef, Ed, ND, T, gD=2):
    # ND+ = ND * (1 - fD)
    # fD = expit(-( (Ed - Ef)/kT + log(gD) ))
    fD = expit(-((Ed - Ef) / (kB*T) + np.log(gD)))
    return ND * (1.0 - fD)

def NDp_raw(Ef, Ed, ND, T, gD=2):
    """
    ND+ = ND * (1 - fD)
    fD = 1 / (1 + gD * exp((Ed - Ef)/kT))
    """
    expo = (Ed - Ef) / (kB * T)
    fD = 1.0 / (1.0 + gD * np.exp(expo))
    return ND * (1.0 - fD)

# -----------------------------
# テスト設定
# -----------------------------

T = 300.0  # K
kBT = kB * T
ND = 1e21  # arbitrary (m^-3)
Ed = 0.05 * q  # 0.05 eV donor level

gD_values = [1, 2, 4, 10]
Efmin, Efmax = -0.5, 0.5
Ef_values = np.linspace(Efmin*q, Efmax*q, 21)

# 許容誤差
tol = 1e-12 * ND   # ND に比例させる（相対誤差 ~1e-12）

# -----------------------------
# 実行
# -----------------------------

print("Testing NDp implementations (expit vs raw FD)")
print("T = 300 K, ND = 1e21 m^-3, Ed = 0.05 eV")
print("diff torelance:", tol)
print("--------------------------------------------------------------")
print(" gD    Ef[eV]     values     diff          result")
print("--------------------------------------------------------------")

for gD in gD_values:
    for Ef in Ef_values:
        val1 = NDp_expit(Ef, Ed, ND, T, gD)
        val2 = NDp_raw(Ef, Ed, ND, T, gD)
        diff = abs(val1 - val2)
        result = "PASS" if diff < tol else "FAIL"

        eta = Ef / kBT
        xD = Ed / kBT
        val3 = ND * ionized_donor_frac(eta, xD, g = 1.0 / gD)
        diff = abs(val1 - val3)
        result = "PASS" if diff < tol and result else "FAIL"

        print(f"{gD:3d}   {Ef/q:7.3f}   {val1:12.5e} - {val2:12.5e} - {val3:12.5e} = {diff:12.5e}   {result}")
    print("--------------------------------------------------------------")
