import numpy as np
import matplotlib.pyplot as plt
from scipy.integrate import quad
from scipy.special import expit, gamma
from functools import lru_cache

@lru_cache(maxsize=4096)
def Fj(j, eta):
    """Fj(j,eta) = ∫_0^∞ x^j / (1+exp(x-eta)) dx  （1/Gamma(j+1) なし）"""
    integrand = lambda x: (x**j) * expit(eta - x)
    res, _ = quad(integrand, 0.0, np.inf, epsabs=1e-10, epsrel=1e-10, limit=300)
    return res

def hall_factor_from_l_exponent(r_l, ef_ev, kt=0.0259):
    """
    r_l : 平均自由行程 l ∝ E^{r_l} の指数（スライドの r）
    非縮退では τ ∝ E^{r_l-1/2} なので、輸送指数 s = r_l - 1/2 を使って Hall 因子を計算する
    """
    eta = ef_ev / kt

    s = r_l - 0.5  # ★ここが本質：tau exponent

    F12    = Fj(0.5, eta)
    Fs12   = Fj(s + 0.5, eta)
    F2s12  = Fj(2.0*s + 0.5, eta)

    pref = (1.5 * (2.0*s + 1.5)) / ((s + 1.5)**2)
    return pref * (F12 * F2s12) / (Fs12**2)

def nondeg_limit_from_l_exponent(r_l):
    """η→-∞ の非縮退極限（確認用）"""
    s = r_l - 0.5
    pref = (1.5 * (2.0*s + 1.5)) / ((s + 1.5)**2)
    return pref * (gamma(1.5) * gamma(2.0*s + 1.5)) / (gamma(s + 1.5)**2)

def main():
    ef_range = np.linspace(-0.2, 0.5, 120)
    r_l_list = [2.0, 1.5, 1.0, 0.5, 0.0]  # これは「lの指数」

    plt.figure(figsize=(8, 6))
    for r_l in r_l_list:
        y = [hall_factor_from_l_exponent(r_l, ef) for ef in ef_range]
        plt.plot(ef_range, y, label=f"r_l = {r_l:.1f}")

        y0 = nondeg_limit_from_l_exponent(r_l)
        plt.axhline(y0, linestyle="--", alpha=0.25)

    plt.xlabel(r"$E_F$ (eV)")
    plt.ylabel(r"$F_H\;(\gamma_{\mathrm{Hall}})$")
    plt.ylim(0.8, 2.2)
    plt.xlim(-0.2, 0.5)
    plt.grid(True, alpha=0.25)
    plt.legend()
    plt.title(r"Hall factor using $\tau\propto E^{r_l-1/2}$ (nondegenerate)")
    plt.show()

if __name__ == "__main__":
    main()
