import sys
import numpy as np
from scipy import integrate
import matplotlib.pyplot as plt


"""
金属の有限温度 T における電子密度 N の計算。
"""


# -----------------------------
# 物理定数（numpy から取得）
# -----------------------------
e = 1.60218e-19        # C
kB = 1.380658e-23      # J/K
me = 9.1093897e-31     # kg
h = 6.6260755e-34      # J s
pi = np.pi


# -----------------------------
# 状態密度 D(E)
# -----------------------------
def density_of_states(E, D0):
    return D0 * np.sqrt(E)


# -----------------------------
# フェルミ分布 f(E)
# -----------------------------
def fermi_dirac(E, T, EF):
    if T == 0.0:
        return 1.0 if E < EF else 0.0

    ekBT = e / kB / T
    return 1.0 / (np.exp((E - EF) * ekBT) + 1.0)


# -----------------------------
# 被積分関数 D(E) f(E)
# -----------------------------
def integrand(E, T, EF, D0):
    return density_of_states(E, D0) * fermi_dirac(E, T, EF)


# -----------------------------
# 電子密度の計算
# -----------------------------
def compute_density(T, EF, nrange, eps, D0):
    ekBT = e / (kB * T)
    dE = nrange * kB * T / e
    Emax = EF + dE

    N, _ = integrate.quad(
        lambda E: integrand(E, T, EF, D0),
        0.0, Emax,
        epsrel=eps
    )
    return N, Emax


# -----------------------------
# プロット
# -----------------------------
def plot_results(T, EF, Emax, D0):
    ekBT = e / (kB * T)

    xE = np.arange(0.0, Emax, 0.01)
    yDe = density_of_states(xE, D0)
    yfe = np.array([fermi_dirac(E, T, EF) for E in xE])
    yDe0 = np.array([integrand(E, 0.0, EF, D0) for E in xE])
    yDefe = np.array([integrand(E, T, EF, D0) for E in xE])

    # f(E) を見やすくするためスケール調整
    yfe_scaled = yfe * np.max(yDefe)

    fig, axis = plt.subplots(figsize=(8, 6))
    fontsize = 14

    axis.plot(xE, yDe, label=r'$D_e(E)$', color='black')
    axis.plot(xE, yDe0, label=r'$D_e(E) f_{FD}(E,0)$', color='blue')
    axis.plot(xE, yfe_scaled, label=r'$f_{FD}(E,T,E_F)$', color='red')
    axis.plot(xE, yDefe, label=r'$D_e(E) f_{FD}(E,T)$', color='darkgreen')

    axis.axvline(EF, linestyle='dashed', color='red', linewidth=0.5)
    axis.axhline(0.0, linestyle='dashed', color='red', linewidth=0.5)
    axis.fill_between(xE, yDefe, 0.0, facecolor='lime', alpha=0.5)

    axis.set_xlabel(r'$E - E_C$ (eV)', fontsize=fontsize)
    axis.set_ylabel(r'DOS (states/cm$^3$)', fontsize=fontsize)
    axis.legend(fontsize=fontsize)
    axis.tick_params(labelsize=fontsize)

    plt.tight_layout()
    plt.show()


# -----------------------------
# メイン処理
# -----------------------------
def main():
    # デフォルト値
    T = 300.0
    EF = 5.0
    nrange = 6.0
    eps = 1e-8

    # コマンドライン引数
    argv = sys.argv
    if len(argv) >= 2:
        T = float(argv[1])
    if len(argv) >= 3:
        EF = float(argv[2])

    # 状態密度の係数
    S = 0.5
    D0 = (2 * S + 1) * 2 * pi * (2 * me)**1.5 / h**3 * e**1.5 * 1e-6

    print(f"\n=== T = 0 K, EF = {EF} eV ===")
    Na = 2.0 / 3.0 * D0 * EF**1.5
    print(f"Analytical: N = {Na:.6e} cm^-3")

    N0, _ = integrate.quad(lambda E: density_of_states(E, D0), 0.0, EF, epsrel=eps)
    print(f"Numerical:  N = {N0:.6e} cm^-3")

    print(f"\n=== T = {T} K, EF = {EF} eV ===")
    N, Emax = compute_density(T, EF, nrange, eps, D0)
    print(f"Numerical:  N = {N:.6e} cm^-3")

    plot_results(T, EF, Emax, D0)


# -----------------------------
# エントリーポイント
# -----------------------------
if __name__ == "__main__":
    main()
