"""
#Prompt #all
半導体のキャリア統計をシミュレートする、以下の仕様を満たすPythonプログラムを作成してください。

1. 【物理モデル】
- 伝導帯、価電子帯、ドナー/アクセプター準位を考慮し、電荷中性条件 (p - n + ND+ - NA- = 0) を満たすフェルミ準位 Ef を決定する。
- キャリア密度 (n, p) は、ボルツマン近似ではなく「フェルミ・ディラック積分 (オーダー1/2)」を用いて計算すること。
- 不純物のイオン化 (ND+, NA-) は、占有関数と退化因子 (gD=2, gA=4) を用いて計算すること。

2. 【数値計算の安定化】
- 強い縮退（Efがバンド内にある状態）でも計算できるよう、指数関数のオーバーフローを避けるために `scipy.special.expit` を使用した数値積分を行うこと。
- Efの決定には `scipy.optimize.brentq` (Brent法) を使用し、解を囲む範囲 (bracket) が不明な場合は自動で符号反転を確認するまで探索範囲を広げるロジックを実装すること。

3. 【プログラム構造】
- `argparse` を使用して、バンドギャップ(Eg)、有効質量(me, mh)、不純物密度(ND, NA)などのパラメータを外部から与えられるようにする。
- `initialize()` 関数でパラメータの初期設定を行い、`main()` で計算とプロットを行う。
- `if __name__ == "__main__":` ギミックを使用すること。
- 各関数名（m2Nc, m2Nv, fe, fh, Fj, Ne, Nh, NDp, NAm, deltaQ）は短く、物理的な意味を反映させること。

4. 【出力とプロット】
- 指定された温度範囲（例: 50K〜600K）でスイープを行い、結果をコンソールに表示する。
- グラフ表示の密度範囲は、Nmin,Maxパラメータで指定できるようにする (デフォルト Nmin=1e10, Nmax=1e22)
- グラフは2軸プロットとし、左軸に「キャリア密度・イオン化不純物密度 (ログスケール)」、右軸に「フェルミ準位 Ef (リニアスケール)」を描画すること。
"""

#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import argparse
import numpy as np
import matplotlib.pyplot as plt

from scipy import integrate, optimize, special
from scipy.constants import k as kB_SI, h as h_SI, m_e as m0_SI, e as q_SI, pi


# --- core physics utilities (short names, physical meaning) -------------------

def m2Nc(me, T):
    """Effective density of states in conduction band Nc [cm^-3].
    me: electron effective mass in units of m0
    """
    m = me * m0_SI
    Nc_m3 = 2.0 * ((2.0 * pi * m * kB_SI * T) / (h_SI ** 2)) ** 1.5
    return Nc_m3 / 1e6


def m2Nv(mh, T):
    """Effective density of states in valence band Nv [cm^-3].
    mh: hole effective mass in units of m0
    """
    m = mh * m0_SI
    Nv_m3 = 2.0 * ((2.0 * pi * m * kB_SI * T) / (h_SI ** 2)) ** 1.5
    return Nv_m3 / 1e6


def fe(E, Ef, kT):
    """Fermi-Dirac occupation for electrons: f(E) = 1/(1+exp((E-Ef)/kT)) (stable)."""
    return special.expit((Ef - E) / kT)


def fh(E, Ef, kT):
    """Hole occupation: 1 - f(E)."""
    return 1.0 - fe(E, Ef, kT)


def Fj(eta):
    """Fermi-Dirac integral of order 1/2:
       F_{1/2}(eta) = (2/sqrt(pi)) * ∫_0^∞ sqrt(eps) / (1+exp(eps-eta)) d eps
    Uses expit to avoid overflow under strong degeneracy.
    """
    def integrand(eps):
        return np.sqrt(eps) * special.expit(eta - eps)

    val, _ = integrate.quad(integrand, 0.0, np.inf, limit=400)
    return (2.0 / np.sqrt(pi)) * val


def Ne(Ec, Ef, T, me, kT):
    """Electron density n [cm^-3] using F_{1/2}."""
    eta = (Ef - Ec) / kT
    return m2Nc(me, T) * Fj(eta)


def Nh(Ev, Ef, T, mh, kT):
    """Hole density p [cm^-3] using F_{1/2}."""
    eta = (Ev - Ef) / kT
    return m2Nv(mh, T) * Fj(eta)


def NDp(ND, Ed, Ef, kT, gD=2.0):
    """Ionized donors ND+ [cm^-3].
    Donor occupancy (neutral) fD = 1/(1 + gD*exp((Ed-Ef)/kT)), so ND+ = ND*(1-fD).
    """
    fD = special.expit((Ef - Ed) / kT - np.log(gD))
    return ND * (1.0 - fD)


def NAm(NA, Ea, Ef, kT, gA=4.0):
    """Ionized acceptors NA- [cm^-3].
    Occupied (negative) fraction fA = 1/(1 + gA*exp((Ea-Ef)/kT)).
    """
    fA = special.expit((Ef - Ea) / kT - np.log(gA))
    return NA * fA


def deltaQ(Ef, T, pset):
    """Charge neutrality residual: p - n + ND+ - NA-."""
    Eg, me, mh, ND, NA, Ed, Ea = pset
    Ev, Ec = 0.0, Eg

    kT = (kB_SI * T) / q_SI  # [eV]
    n = Ne(Ec, Ef, T, me, kT)
    p = Nh(Ev, Ef, T, mh, kT)
    ndp = NDp(ND, Ed, Ef, kT)
    nam = NAm(NA, Ea, Ef, kT)
    return p - n + ndp - nam


def _bracket_root(func, x0, step0, grow, max_iter):
    """Expand [a,b] until func(a)*func(b) < 0."""
    a = x0 - step0
    b = x0 + step0
    fa = func(a)
    fb = func(b)

    step = step0
    for _ in range(max_iter):
        if fa == 0.0:
            return a, a
        if fb == 0.0:
            return b, b
        if fa * fb < 0.0:
            return a, b

        step *= grow
        a = x0 - step
        b = x0 + step
        fa = func(a)
        fb = func(b)

    raise RuntimeError("Failed to bracket Ef (no sign change). Try widening search or check parameters.")


def find_Ef(T, pset, bracket_halfwidth_eV=0.5):
    """Solve deltaQ(Ef)=0 via brentq with automatic bracketing expansion."""
    Eg, *_ = pset
    Ev, Ec = 0.0, Eg
    x0 = 0.5 * (Ev + Ec)

    def f(Ef):
        return deltaQ(Ef, T, pset)

    a, b = _bracket_root(
        func=f,
        x0=x0,
        step0=bracket_halfwidth_eV,
        grow=1.7,
        max_iter=60,
    )
    if a == b:
        return a
    return optimize.brentq(f, a, b, xtol=1e-10, rtol=1e-10, maxiter=200)


# --- program structure --------------------------------------------------------

def initialize():
    p = argparse.ArgumentParser(description="Degenerate semiconductor carrier statistics with FD integral (order 1/2).")

    p.add_argument("--Eg", type=float, default=1.12, help="Bandgap Eg [eV], with Ev=0 and Ec=Eg.")
    p.add_argument("--me", type=float, default=1.08, help="Electron effective mass (relative to m0).")
    p.add_argument("--mh", type=float, default=0.81, help="Hole effective mass (relative to m0).")

    p.add_argument("--ND", type=float, default=1e16, help="Donor density ND [cm^-3].")
    p.add_argument("--NA", type=float, default=0.0, help="Acceptor density NA [cm^-3].")

    p.add_argument("--Ed_bind", type=float, default=0.045, help="Donor binding energy from Ec: Ed = Ec - Ed_bind [eV].")
    p.add_argument("--Ea_bind", type=float, default=0.045, help="Acceptor binding energy from Ev: Ea = Ev + Ea_bind [eV].")

    p.add_argument("--Tmin", type=float, default=50.0, help="Minimum temperature [K].")
    p.add_argument("--Tmax", type=float, default=600.0, help="Maximum temperature [K].")
    p.add_argument("--Tnum", type=int, default=56, help="Number of temperature points (inclusive sweep).")

    p.add_argument("--Nmin", type=float, default=1e10, help="Plot y-min for densities [cm^-3].")
    p.add_argument("--Nmax", type=float, default=1e22, help="Plot y-max for densities [cm^-3].")

    p.add_argument("--bracket", type=float, default=0.5, help="Initial half-width for Ef bracketing [eV].")
    p.add_argument("--no-plot", action="store_true", help="Disable plotting.")

    return p.parse_args()


def main():
    args = initialize()

    Ev, Ec = 0.0, args.Eg
    Ed = Ec - args.Ed_bind
    Ea = Ev + args.Ea_bind

    Ts = np.linspace(args.Tmin, args.Tmax, args.Tnum)

    pset = (args.Eg, args.me, args.mh, args.ND, args.NA, Ed, Ea)

    out = []
    print("#  T[K]      Ef[eV]          n[cm^-3]          p[cm^-3]         ND+[cm^-3]        NA-[cm^-3]")
    for T in Ts:
        kT = (kB_SI * T) / q_SI
        Ef = find_Ef(T, pset, bracket_halfwidth_eV=args.bracket)

        n = Ne(Ec, Ef, T, args.me, kT)
        p_ = Nh(Ev, Ef, T, args.mh, kT)
        ndp = NDp(args.ND, Ed, Ef, kT)
        nam = NAm(args.NA, Ea, Ef, kT)

        out.append((T, Ef, n, p_, ndp, nam))
        print(f"{T:7.1f}  {Ef:10.6f}  {n:14.6e}  {p_:14.6e}  {ndp:14.6e}  {nam:14.6e}")

    if args.no_plot:
        return

    out = np.array(out, dtype=float)
    T = out[:, 0]
    Ef = out[:, 1]
    n = out[:, 2]
    p_ = out[:, 3]
    ndp = out[:, 4]
    nam = out[:, 5]

    plt.style.use("seaborn-v0_8")
    fig, axL = plt.subplots(figsize=(10, 6))
    axR = axL.twinx()

    axL.semilogy(T, n, label="n", lw=2)
    axL.semilogy(T, p_, label="p", lw=2)
    axL.semilogy(T, ndp, label="ND+", lw=2)
    axL.semilogy(T, nam, label="NA-", lw=2)

    axR.plot(T, Ef, label="Ef", color="black", ls="--", lw=2)

    axL.set_xlabel("Temperature [K]")
    axL.set_ylabel("Carrier / Ionized impurity density [cm$^{-3}$] (log)")
    axR.set_ylabel("Fermi level Ef [eV] (linear)")

    axL.set_ylim(args.Nmin, args.Nmax)
    axR.set_ylim(Ev - 0.2, Ec + 0.2)

    h1, l1 = axL.get_legend_handles_labels()
    h2, l2 = axR.get_legend_handles_labels()
    axL.legend(h1 + h2, l1 + l2, loc="best")

    axL.grid(True, which="both", ls=":", alpha=0.5)
    plt.title("Degenerate carrier statistics with FD integral (order 1/2)")
    plt.tight_layout()
    plt.show()


if __name__ == "__main__":
    main()
