"""
#Prompt #all
半導体のキャリア統計をシミュレートする、以下の仕様を満たすPythonプログラムを作成してください。

1. 【物理モデル】
- 伝導帯、価電子帯、ドナー/アクセプター準位を考慮し、電荷中性条件 (p - n + ND+ - NA- = 0) を満たすフェルミ準位 Ef を決定する。
- キャリア密度 (n, p) は、ボルツマン近似ではなく「フェルミ・ディラック積分 (オーダー1/2)」を用いて計算すること。
- 不純物のイオン化 (ND+, NA-) は、占有関数と退化因子 (gD=2, gA=4) を用いて計算すること。

2. 【数値計算の安定化】
- 強い縮退（Efがバンド内にある状態）でも計算できるよう、指数関数のオーバーフローを避けるために `scipy.special.expit` を使用した数値積分を行うこと。
- Efの決定には `scipy.optimize.brentq` (Brent法) を使用し、解を囲む範囲 (bracket) が不明な場合は自動で符号反転を確認するまで探索範囲を広げるロジックを実装すること。

3. 【プログラム構造】
- `argparse` を使用して、バンドギャップ(Eg)、有効質量(me, mh)、不純物密度(ND, NA)などのパラメータを外部から与えられるようにする。
- `initialize()` 関数でパラメータの初期設定を行い、`main()` で計算とプロットを行う。
- `if __name__ == "__main__":` ギミックを使用すること。
- 各関数名（m2Nc, m2Nv, fe, fh, Fj, Ne, Nh, NDp, NAm, deltaQ）は短く、物理的な意味を反映させること。

4. 【出力とプロット】
- 指定された温度範囲（例: 50K〜600K）でスイープを行い、結果をコンソールに表示する。
- グラフ表示の密度範囲は、Nmin,Maxパラメータで指定できるようにする (デフォルト Nmin=1e10, Nmax=1e22)
- グラフは2軸プロットとし、左軸に「キャリア密度・イオン化不純物密度 (ログスケール)」、右軸に「フェルミ準位 Ef (リニアスケール)」を描画すること。
"""

#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
半導体キャリア統計シミュレーション
- フェルミ・ディラック積分（オーダー1/2）を数値積分で評価
- scipy.special.expit を用いて数値安定化
- フェルミ準位 Ef を電荷中性条件で brentq により決定（自動ブラケット拡張）
- コマンドライン引数で物性値を指定可能
"""

#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
修正版: 半導体キャリア統計シミュレーション
- フェルミ・ディラック積分 (order 1/2) を無次元積分で評価 (expit を使用)
- Nc, Nv を解析式で計算
- 不純物占有は縮退因子を考慮し数値安定化
- Ef は brentq で求め、ブラケット自動拡張を実装
"""

import argparse
import numpy as np
from scipy import integrate, optimize, special
import matplotlib.pyplot as plt
import math

# 基本定数
m0 = 9.10938356e-31       # kg
h = 6.62607015e-34        # Planck constant J s
hbar = h / (2.0 * math.pi)
q = 1.602176634e-19       # C
kB = 1.380649e-23         # J/K
kB_eV = 8.617333262145e-5 # eV/K

# -------------------------
# 短い関数名で物理的意味を反映
# -------------------------

def m2Nc_T(m_eff, T):
    """
    有効状態密度 Nc(T) (SI: m^-3)
    Nc = 2 * (2*pi*m_e*kB*T / h^2)^{3/2}
    m_eff: kg
    """
    return 2.0 * (2.0 * math.pi * m_eff * kB * T / (h**2))**1.5

def m2Nv_T(m_eff, T):
    """Nv(T) (SI: m^-3)"""
    return m2Nc_T(m_eff, T)

def F_half(eta):
    """
    フェルミ・ディラック積分 F_{1/2}(eta)
    F_{1/2}(eta) = (2/sqrt(pi)) * ∫_0^∞ sqrt(eps) / (1 + exp(eps - eta)) deps
    数値的に安定化して expit を使用: 1/(1+exp(eps-eta)) = expit(eta - eps)
    """
    # integrand uses expit for stability
    def integrand(eps):
        if eps == 0.0:
            return 0.0
        return math.sqrt(eps) * special.expit(eta - eps)
    # integrate from 0 to inf
    val, err = integrate.quad(integrand, 0.0, np.inf, epsabs=0.0, epsrel=1e-6, limit=200)
    return (2.0 / math.sqrt(math.pi)) * val

def fe_occ(E, Ef, T):
    """フェルミ占有 f(E) を安定に評価"""
    return special.expit(-(E - Ef) / (kB * T))

def Ne(Ec, Ef, T, meff):
    """
    電子密度 n = Nc(T) * F_{1/2}((Ef - Ec)/kT)
    Ec, Ef: J
    meff: kg
    """
    eta = (Ef - Ec) / (kB * T)
    Nc = m2Nc_T(meff, T)
    return Nc * F_half(eta)

def Nh(Ev, Ef, T, mhff):
    """
    正孔密度 p = Nv(T) * F_{1/2}((Ev - Ef)/kT)
    注意: holes use eta_h = (Ev - Ef)/kT
    """
    eta_h = (Ev - Ef) / (kB * T)
    Nv = m2Nv_T(mhff, T)
    return Nv * F_half(eta_h)

def NDp(E_D, Ef, T, ND, gD=2.0):
    """
    ドナーの正にイオン化 ND+ = ND * (1 - f_D)
    f_D = 1 / (1 + gD * exp((E_D - Ef)/kT))
    数値安定化: expo = (E_D - Ef)/kT
    """
    expo = (E_D - Ef) / (kB * T)
    # 安定評価
    if expo > 700:
        fD = 0.0
    elif expo < -700:
        # exp(expo) ~ 0 -> denom ~ 1 -> fD ~ 1
        fD = 1.0 / (1.0 + gD * 0.0)
    else:
        denom = 1.0 + gD * math.exp(expo)
        fD = 1.0 / denom
    return ND * (1.0 - fD)

def NAm(E_A, Ef, T, NA, gA=4.0):
    """
    アクセプタの負にイオン化 NA- = NA * f_A
    f_A = 1 / (1 + gA * exp((Ef - E_A)/kT))
    """
    expo = (Ef - E_A) / (kB * T)
    if expo > 700:
        fA = 0.0
    elif expo < -700:
        fA = 1.0 / (1.0 + gA * 0.0)
    else:
        denom = 1.0 + gA * math.exp(expo)
        fA = 1.0 / denom
    return NA * fA

def deltaQ(Ef, params):
    """
    電荷中性: p - n + ND+ - NA-
    すべて SI 単位 (m^-3)
    """
    Ec = params['Ec']
    Ev = params['Ev']
    T = params['T']
    meff = params['meff']
    mhff = params['mhff']
    ND = params['ND']
    NA = params['NA']
    E_D = params['E_D']
    E_A = params['E_A']
    gD = params['gD']
    gA = params['gA']

    n = Ne(Ec, Ef, T, meff)
    p = Nh(Ev, Ef, T, mhff)
    ndp = NDp(E_D, Ef, T, ND, gD)
    nam = NAm(E_A, Ef, T, NA, gA)
    return p - n + ndp - nam

# -------------------------
# Ef を求める: 自動ブラケット拡張
# -------------------------

def find_Ef(params):
    Ec = params['Ec']
    Ev = params['Ev']
    T = params['T']

    # 初期ブラケット: Ev - 1 eV から Ec + 1 eV (J)
    span_eV = 1.0
    left = Ev - span_eV * q
    right = Ec + span_eV * q
    f_left = deltaQ(left, params)
    f_right = deltaQ(right, params)

    # もし符号反転がなければ指数的に拡張（最大回数制限）
    max_iter = 60
    i = 0
    while f_left * f_right > 0 and i < max_iter:
        # 広げる幅は温度スケールにも依存させる（kT ~）
        factor = 2.0 ** i
        left = Ev - (span_eV * factor) * q
        right = Ec + (span_eV * factor) * q
        f_left = deltaQ(left, params)
        f_right = deltaQ(right, params)
        i += 1

    if f_left * f_right > 0:
        raise RuntimeError("フェルミ準位のブラケットが見つかりません。パラメータを確認してください。")

    Ef = optimize.brentq(lambda E: deltaQ(E, params), left, right, xtol=1e-12, rtol=1e-8, maxiter=200)
    return Ef

# -------------------------
# CLI 初期化と main
# -------------------------

def initialize():
    p = argparse.ArgumentParser(description="半導体キャリア統計シミュレーション（修正版）")
    p.add_argument("--Eg", type=float, default=1.12, help="バンドギャップ Eg (eV)")
    p.add_argument("--me", type=float, default=1.08, help="伝導帯有効質量 me*/m0")
    p.add_argument("--mh", type=float, default=0.56, help="価電子帯有効質量 mh*/m0")
    p.add_argument("--ND", type=float, default=1e16, help="ドナー密度 ND (cm^-3)")
    p.add_argument("--NA", type=float, default=0.0, help="アクセプタ密度 NA (cm^-3)")
    p.add_argument("--ED", type=float, default=0.05, help="ドナー準位 Ec - ED (eV)")
    p.add_argument("--EA", type=float, default=0.05, help="アクセプタ準位 EA - Ev (eV)")
    p.add_argument("--gD", type=float, default=2.0, help="ドナー縮退因子")
    p.add_argument("--gA", type=float, default=4.0, help="アクセプタ縮退因子")
    p.add_argument("--Tmin", type=float, default=50.0, help="温度最小 (K)")
    p.add_argument("--Tmax", type=float, default=600.0, help="温度最大 (K)")
    p.add_argument("--Tsteps", type=int, default=30, help="温度ステップ数")
    p.add_argument("--Nmin", type=float, default=1e10, help="プロット下限 (cm^-3)")
    p.add_argument("--Nmax", type=float, default=1e22, help="プロット上限 (cm^-3)")
    p.add_argument("--verbose", action="store_true", help="詳細出力")
    return p.parse_args()

def main():
    args = initialize()

    # パラメータ変換
    Eg = args.Eg
    meff = args.me * m0
    mhff = args.mh * m0
    ND = args.ND * 1e6   # cm^-3 -> m^-3
    NA = args.NA * 1e6
    Ev_eV = 0.0
    Ec_eV = Eg
    Ev = Ev_eV * q
    Ec = Ec_eV * q
    E_D = (Ec_eV - args.ED) * q
    E_A = (Ev_eV + args.EA) * q
    gD = args.gD
    gA = args.gA

    Ts = np.linspace(args.Tmin, args.Tmax, args.Tsteps)
    Ns = []
    Ps = []
    NDps = []
    NAmS = []
    Efs = []

    print("T[K]    n[cm^-3]      p[cm^-3]      ND+[cm^-3]    NA-[cm^-3]    Ef[eV]")
    for T in Ts:
        params = {
            'Ec': Ec, 'Ev': Ev, 'T': T,
            'meff': meff, 'mhff': mhff,
            'ND': ND, 'NA': NA,
            'E_D': E_D, 'E_A': E_A,
            'gD': gD, 'gA': gA
        }
        try:
            Ef_J = find_Ef(params)
        except Exception as e:
            print(f"T={T:.1f} K: Ef を求められませんでした: {e}")
            # fallback: mid-gap
            Ef_J = (Ec + Ev) / 2.0

        n = Ne(Ec, Ef_J, T, meff)
        p = Nh(Ev, Ef_J, T, mhff)
        ndp = NDp(E_D, Ef_J, T, ND, gD)
        nam = NAm(E_A, Ef_J, T, NA, gA)

        Ns.append(n)
        Ps.append(p)
        NDps.append(ndp)
        NAmS.append(nam)
        Efs.append(Ef_J / q)

        print(f"{T:6.1f}  {n*1e-6:12.4e}  {p*1e-6:12.4e}  {ndp*1e-6:12.4e}  {nam*1e-6:12.4e}  {Ef_J/q:8.4f}")

    # プロット
    fig, ax1 = plt.subplots(figsize=(8,6))
    ax2 = ax1.twinx()

    Ns_cm = np.array(Ns) * 1e-6
    Ps_cm = np.array(Ps) * 1e-6
    NDps_cm = np.array(NDps) * 1e-6
    NAmS_cm = np.array(NAmS) * 1e-6

    ax1.set_yscale('log')
    ax1.set_ylim(args.Nmin, args.Nmax)
    ax1.plot(Ts, Ns_cm, 'b-o', label='n (electrons)')
    ax1.plot(Ts, Ps_cm, 'r-s', label='p (holes)')
    if np.any(NDps_cm > 0):
        ax1.plot(Ts, NDps_cm, 'g--', label='ND+')
    if np.any(NAmS_cm > 0):
        ax1.plot(Ts, NAmS_cm, 'm--', label='NA-')

    ax1.set_xlabel("Temperature (K)")
    ax1.set_ylabel("Density (cm^-3) [log scale]")
    ax1.grid(True, which='both', ls='--', alpha=0.5)
    ax1.legend(loc='upper left')

    ax2.plot(Ts, Efs, 'k-', linewidth=2, label='Ef (eV)')
    ax2.set_ylabel("Fermi level Ef (eV) [linear]")
    ax2.legend(loc='upper right')

    plt.title(f"Carrier statistics: Eg={Eg} eV, ND={args.ND:.3g} cm^-3, NA={args.NA:.3g} cm^-3")
    plt.tight_layout()
    plt.show()

if __name__ == "__main__":
    main()
