"""
#Prompt #all
半導体のキャリア統計をシミュレートする、以下の仕様を満たすPythonプログラムを作成してください。

1. 【物理モデル】
- 伝導帯、価電子帯、ドナー/アクセプター準位を考慮し、電荷中性条件 (p - n + ND+ - NA- = 0) を満たすフェルミ準位 Ef を決定する。
- キャリア密度 (n, p) は、ボルツマン近似ではなく「フェルミ・ディラック積分 (オーダー1/2)」を用いて計算すること。
- 不純物のイオン化 (ND+, NA-) は、占有関数と退化因子 (gD=2, gA=4) を用いて計算すること。

2. 【数値計算の安定化】
- 強い縮退（Efがバンド内にある状態）でも計算できるよう、指数関数のオーバーフローを避けるために `scipy.special.expit` を使用した数値積分を行うこと。
- Efの決定には `scipy.optimize.brentq` (Brent法) を使用し、解を囲む範囲 (bracket) が不明な場合は自動で符号反転を確認するまで探索範囲を広げるロジックを実装すること。

3. 【プログラム構造】
- `argparse` を使用して、バンドギャップ(Eg)、有効質量(me, mh)、不純物密度(ND, NA)などのパラメータを外部から与えられるようにする。
- `initialize()` 関数でパラメータの初期設定を行い、`main()` で計算とプロットを行う。
- `if __name__ == "__main__":` ギミックを使用すること。
- 各関数名（m2Nc, m2Nv, fe, fh, Fj, Ne, Nh, NDp, NAm, deltaQ）は短く、物理的な意味を反映させること。

4. 【出力とプロット】
- 指定された温度範囲（例: 50K〜600K）でスイープを行い、結果をコンソールに表示する。
- グラフ表示の密度範囲は、Nmin,Maxパラメータで指定できるようにする (デフォルト Nmin=1e10, Nmax=1e22)
- グラフは2軸プロットとし、左軸に「キャリア密度・イオン化不純物密度 (ログスケール)」、右軸に「フェルミ準位 Ef (リニアスケール)」を描画すること。
"""

#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
半導体キャリア統計シミュレーション
- フェルミ・ディラック積分（オーダー1/2）を数値積分で評価
- scipy.special.expit を用いて数値安定化
- フェルミ準位 Ef を電荷中性条件で brentq により決定（自動ブラケット拡張）
- コマンドライン引数で物性値を指定可能
"""

import argparse
import numpy as np
import math
from scipy import integrate, optimize, special
import matplotlib.pyplot as plt

# 基本定数（SI）
m0 = 9.10938356e-31        # electron mass (kg)
hbar = 1.054571817e-34     # reduced Planck constant (J s)
q = 1.602176634e-19        # elementary charge (C)
kB = 1.380649e-23          # Boltzmann constant (J/K)
kB_eV = 8.617333262145e-5  # Boltzmann constant (eV/K)

# ---------------------------
# 関数名は指定に合わせて短く、物理的意味を反映
# ---------------------------

def m2Nc(m_eff):
    """
    DOS 前因子 (conduction/valence) のエネルギー依存を除いた係数
    g_c(E) = A_c * sqrt(E - Ec)  (E in J)
    A_c = 1/(2*pi^2) * (2*m*/hbar^2)^{3/2}
    """
    A = 1.0 / (2.0 * math.pi**2) * (2.0 * m_eff / hbar**2)**1.5
    return A

def m2Nv(m_eff):
    """同上（価電子帯用）"""
    return m2Nc(m_eff)

def fe(E, Ef, T):
    """
    フェルミ分布 f(E) を数値安定化して返す。
    E, Ef: エネルギー（J）
    T: 温度（K）
    使用: expit( - (E - Ef) / (kB T) ) = 1/(1+exp((E-Ef)/kT))
    """
    arg = - (E - Ef) / (kB * T)
    return special.expit(arg)

def fh(E, Ef, T):
    """ホール（正孔）占有確率 1 - f(E)"""
    return 1.0 - fe(E, Ef, T)

def Fj(x, prefactor, Ef, band_edge, T):
    """
    積分の被積分関数（エネルギー差 x = E - band_edge >= 0 を想定）
    g(E) = prefactor * sqrt(x)
    f(E) = fe(band_edge + x, Ef, T)
    戻り値: g(E) * f(E)
    """
    E = band_edge + x
    return prefactor * math.sqrt(x) * fe(E, Ef, T)

def Ne(Ec, Ef, T, m_eff):
    """
    電子密度 n = ∫_{Ec}^{∞} g_c(E) f(E) dE
    変数変換なしで quad を使う（上限は inf）
    Ec, Ef: J
    m_eff: kg
    """
    A = m2Nc(m_eff)
    # integrand for x = E - Ec
    def integrand(x):
        # x may be 0 at lower limit; ensure sqrt(0)=0 handled
        if x <= 0:
            return 0.0
        return Fj(x, A, Ef, Ec, T)
    # integrate from 0 to inf
    val, err = integrate.quad(integrand, 0.0, np.inf, epsabs=0.0, epsrel=1e-6, limit=200)
    return val  # in units of 1/(m^3)

def Nh(Ev, Ef, T, m_eff):
    """
    正孔密度 p = ∫_{-∞}^{Ev} g_v(E) [1 - f(E)] dE
    変数変換: x = Ev - E >= 0
    g_v(E) = A_v * sqrt(Ev - E) = A_v * sqrt(x)
    """
    A = m2Nv(m_eff)
    def integrand(x):
        if x <= 0:
            return 0.0
        E = Ev - x
        # 1 - f(E) = fh(E)
        arg = - (E - Ef) / (kB * T)
        # use expit for stability: 1 - f = 1 - expit(arg) = expit(-arg)
        return A * math.sqrt(x) * fh(E, Ef, T)
    val, err = integrate.quad(integrand, 0.0, np.inf, epsabs=0.0, epsrel=1e-6, limit=200)
    return val

def NDp(E_D, Ef, T, ND, gD=2.0):
    """
    ドナーの正にイオン化した密度 ND+ = ND * (1 - f_D)
    ドナー占有確率 f_D = 1 / (1 + gD * exp((E_D - Ef)/kT))
    E_D, Ef: J
    """
    # use expit for stability: f_D = 1 / (1 + gD * exp((E_D - Ef)/kT))
    # compute exponent safely
    expo = (E_D - Ef) / (kB * T)
    # avoid overflow by using expit-like algebra:
    # f_D = 1 / (1 + gD * exp(expo)) = exp(-log(1 + gD*exp(expo)))
    # but simpler: compute denom = 1 + gD * exp(expo)
    # use np.exp with clipping
    try:
        denom = 1.0 + gD * math.exp(expo)
    except OverflowError:
        denom = math.inf
    fD = 1.0 / denom
    return ND * (1.0 - fD)

def NAm(E_A, Ef, T, NA, gA=4.0):
    """
    アクセプタの負にイオン化した密度 NA- = NA * f_A
    アクセプタ占有確率 f_A = 1 / (1 + gA * exp((Ef - E_A)/kT))
    """
    expo = (Ef - E_A) / (kB * T)
    try:
        denom = 1.0 + gA * math.exp(expo)
    except OverflowError:
        denom = math.inf
    fA = 1.0 / denom
    return NA * fA

def deltaQ(Ef, params):
    """
    電荷中性条件の差分: p - n + ND+ - NA-  (単位: 1/m^3)
    Ef: J
    params: dict に必要パラメータを格納
    """
    Ec = params['Ec']
    Ev = params['Ev']
    T = params['T']
    meff = params['meff']
    mhff = params['mhff']
    ND = params['ND']
    NA = params['NA']
    E_D = params['E_D']
    E_A = params['E_A']
    # compute n, p
    n = Ne(Ec, Ef, T, meff)
    p = Nh(Ev, Ef, T, mhff)
    ndp = NDp(E_D, Ef, T, ND, params['gD'])
    nam = NAm(E_A, Ef, T, NA, params['gA'])
    return p - n + ndp - nam

# ---------------------------
# ブラケット自動拡張と Ef 解法
# ---------------------------

def find_Ef(params, Ef_guess=None):
    """
    brentq を用いて Ef を求める。ブラケットが不明な場合は自動拡張して符号反転を探す。
    エネルギー単位は J。
    """
    Ec = params['Ec']
    Ev = params['Ev']
    # 初期範囲: Ev - 0.5 eV から Ec + 0.5 eV
    span_eV = 0.5
    left = Ev - span_eV * q
    right = Ec + span_eV * q
    f_left = deltaQ(left, params)
    f_right = deltaQ(right, params)
    max_expand = 60  # 最大拡張回数（指数的に広げる）
    expand_factor = 2.0
    i = 0
    while f_left * f_right > 0 and i < max_expand:
        # 拡張
        span = span_eV * (expand_factor**i)
        left = Ev - span * q
        right = Ec + span * q
        f_left = deltaQ(left, params)
        f_right = deltaQ(right, params)
        i += 1
    if f_left * f_right > 0:
        raise RuntimeError("フェルミ準位のブラケットが見つかりませんでした。初期範囲を広げてください。")
    # brentq で解く
    Ef_sol = optimize.brentq(lambda Ef: deltaQ(Ef, params), left, right, xtol=1e-12, rtol=1e-8, maxiter=200)
    return Ef_sol

# ---------------------------
# 初期化と main
# ---------------------------

def initialize():
    p = argparse.ArgumentParser(description="半導体キャリア統計シミュレーション")
    p.add_argument("--Eg", type=float, default=1.12, help="バンドギャップ Eg (eV), default 1.12")
    p.add_argument("--me", type=float, default=1.08, help="伝導帯有効質量 me* / m0 (相対値), default 1.08")
    p.add_argument("--mh", type=float, default=0.56, help="価電子帯有効質量 mh* / m0 (相対値), default 0.56")
    p.add_argument("--ND", type=float, default=1e16, help="ドナー密度 ND (cm^-3)")
    p.add_argument("--NA", type=float, default=0.0, help="アクセプタ密度 NA (cm^-3)")
    p.add_argument("--ED", type=float, default=0.05, help="ドナー準位 Ec - ED (eV) (ドナーが伝導帯に近い場合は小さく)")
    p.add_argument("--EA", type=float, default=0.05, help="アクセプタ準位 EA - Ev (eV)")
    p.add_argument("--gD", type=float, default=2.0, help="ドナー縮退因子 gD (default 2)")
    p.add_argument("--gA", type=float, default=4.0, help="アクセプタ縮退因子 gA (default 4)")
    p.add_argument("--Tmin", type=float, default=50.0, help="温度範囲最小 (K)")
    p.add_argument("--Tmax", type=float, default=600.0, help="温度範囲最大 (K)")
    p.add_argument("--Tsteps", type=int, default=30, help="温度ステップ数")
    p.add_argument("--Nmin", type=float, default=1e10, help="プロット下限 (cm^-3)")
    p.add_argument("--Nmax", type=float, default=1e22, help="プロット上限 (cm^-3)")
    p.add_argument("--verbose", action="store_true", help="詳細出力を表示")
    args = p.parse_args()
    return args

def main():
    args = initialize()

    # パラメータ変換と設定
    Eg = args.Eg
    meff = args.me * m0
    mhff = args.mh * m0
    ND = args.ND * 1e6  # cm^-3 -> m^-3
    NA = args.NA * 1e6
    # バンドエッジ（エネルギー基準: Ev = 0, Ec = Eg）
    Ev_eV = 0.0
    Ec_eV = Eg
    Ev = Ev_eV * q
    Ec = Ec_eV * q
    # 不純物準位（ユーザ指定は Ec - ED for donor, EA - Ev for acceptor）
    E_D = (Ec_eV - args.ED) * q
    E_A = (Ev_eV + args.EA) * q
    gD = args.gD
    gA = args.gA

    # 温度スイープ
    Ts = np.linspace(args.Tmin, args.Tmax, args.Tsteps)
    Ns = []   # electron density (m^-3)
    Ps = []   # hole density (m^-3)
    NDps = []
    NAmS = []
    Efs = []

    print("T[K]    n[cm^-3]      p[cm^-3]      ND+[cm^-3]    NA-[cm^-3]    Ef[eV]")
    for T in Ts:
        params = {
            'Ec': Ec, 'Ev': Ev, 'T': T,
            'meff': meff, 'mhff': mhff,
            'ND': ND, 'NA': NA,
            'E_D': E_D, 'E_A': E_A,
            'gD': gD, 'gA': gA
        }
        try:
            Ef_J = find_Ef(params)
        except Exception as e:
            print(f"温度 {T} K で Ef を求められませんでした: {e}")
            Ef_J = (Ec + Ev) / 2.0

        # densities
        n = Ne(Ec, Ef_J, T, meff)
        p = Nh(Ev, Ef_J, T, mhff)
        ndp = NDp(E_D, Ef_J, T, ND, gD)
        nam = NAm(E_A, Ef_J, T, NA, gA)

        Ns.append(n)
        Ps.append(p)
        NDps.append(ndp)
        NAmS.append(nam)
        Efs.append(Ef_J / q)  # eV

        # print (convert to cm^-3)
        print(f"{T:6.1f}  {n*1e-6:12.4e}  {p*1e-6:12.4e}  {ndp*1e-6:12.4e}  {nam*1e-6:12.4e}  {Ef_J/q:8.4f}")

    # プロット
    fig, ax1 = plt.subplots(figsize=(8,6))
    ax2 = ax1.twinx()

    # convert to cm^-3 for plotting
    Ns_cm = np.array(Ns) * 1e-6
    Ps_cm = np.array(Ps) * 1e-6
    NDps_cm = np.array(NDps) * 1e-6
    NAmS_cm = np.array(NAmS) * 1e-6

    ax1.set_yscale('log')
    ax1.set_ylim(args.Nmin, args.Nmax)
    ax1.plot(Ts, Ns_cm, 'b-o', label='n (electrons)')
    ax1.plot(Ts, Ps_cm, 'r-s', label='p (holes)')
    if np.any(NDps_cm > 0):
        ax1.plot(Ts, NDps_cm, 'g--', label='ND+')
    if np.any(NAmS_cm > 0):
        ax1.plot(Ts, NAmS_cm, 'm--', label='NA-')

    ax1.set_xlabel("Temperature (K)")
    ax1.set_ylabel("Density (cm^-3) [log scale]")
    ax1.grid(True, which='both', ls='--', alpha=0.5)
    ax1.legend(loc='upper left')

    ax2.plot(Ts, Efs, 'k-', linewidth=2, label='Ef (eV)')
    ax2.set_ylabel("Fermi level Ef (eV) [linear]")
    ax2.legend(loc='upper right')

    plt.title(f"Carrier statistics: Eg={Eg} eV, ND={args.ND:.3g} cm^-3, NA={args.NA:.3g} cm^-3")
    plt.tight_layout()
    plt.show()

if __name__ == "__main__":
    main()
