"""
#Prompt #all
半導体のキャリア統計をシミュレートする、以下の仕様を満たすPythonプログラムを作成してください。

1. 【物理モデル】
- 伝導帯、価電子帯、ドナー/アクセプター準位を考慮し、電荷中性条件 (p - n + ND+ - NA- = 0) を満たすフェルミ準位 Ef を決定する。
- キャリア密度 (n, p) は、ボルツマン近似ではなく「フェルミ・ディラック積分 (オーダー1/2)」を用いて計算すること。
- 不純物のイオン化 (ND+, NA-) は、占有関数と退化因子 (gD=2, gA=4) を用いて計算すること。

2. 【数値計算の安定化】
- 強い縮退（Efがバンド内にある状態）でも計算できるよう、指数関数のオーバーフローを避けるために `scipy.special.expit` を使用した数値積分を行うこと。
- Efの決定には `scipy.optimize.brentq` (Brent法) を使用し、解を囲む範囲 (bracket) が不明な場合は自動で符号反転を確認するまで探索範囲を広げるロジックを実装すること。

3. 【プログラム構造】
- `argparse` を使用して、バンドギャップ(Eg)、有効質量(me, mh)、不純物密度(ND, NA)などのパラメータを外部から与えられるようにする。
- `initialize()` 関数でパラメータの初期設定を行い、`main()` で計算とプロットを行う。
- `if __name__ == "__main__":` ギミックを使用すること。
- 各関数名（m2Nc, m2Nv, fe, fh, Fj, Ne, Nh, NDp, NAm, deltaQ）は短く、物理的な意味を反映させること。

4. 【出力とプロット】
- 指定された温度範囲（例: 50K〜600K）でスイープを行い、結果をコンソールに表示する。
- グラフ表示の密度範囲は、Nmin,Maxパラメータで指定できるようにする (デフォルト Nmin=1e10, Nmax=1e22)
- グラフは2軸プロットとし、左軸に「キャリア密度・イオン化不純物密度 (ログスケール)」、右軸に「フェルミ準位 Ef (リニアスケール)」を描画すること。
"""

#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
Semiconductor carrier statistics with full Fermi–Dirac integrals
"""

import argparse
import numpy as np
import matplotlib.pyplot as plt

from scipy.constants import k, h, m_e, e
from scipy.special import expit
from scipy.integrate import quad
from scipy.optimize import brentq


# ============================================================
# Density of states
# ============================================================
def m2Nc(m, T):
    """Effective density of states for conduction band"""
    return 2.0 * ((2 * np.pi * m * m_e * k * T) / h**2) ** (3 / 2)


def m2Nv(m, T):
    """Effective density of states for valence band"""
    return 2.0 * ((2 * np.pi * m * m_e * k * T) / h**2) ** (3 / 2)


# ============================================================
# Fermi–Dirac occupation
# ============================================================
def fe(E, Ef, T):
    return expit((Ef - E) / (k * T))


def fh(E, Ef, T):
    return expit((E - Ef) / (k * T))


# ============================================================
# Fermi–Dirac integral (order 1/2)
# ============================================================
def Fj(eta):
    """Fermi–Dirac integral F_{1/2}(eta)"""
    integrand = lambda x: np.sqrt(x) * expit(eta - x)
    val, _ = quad(integrand, 0, np.inf, limit=200)
    return val


# ============================================================
# Carrier densities
# ============================================================
def Ne(Ef, Ec, T, Nc):
    eta = (Ef - Ec) / (k * T)
    return Nc * Fj(eta)


def Nh(Ef, Ev, T, Nv):
    eta = (Ev - Ef) / (k * T)
    return Nv * Fj(eta)


# ============================================================
# Ionized impurities
# ============================================================
def NDp(Ef, ED, T, ND, gD=2):
    return ND * expit((Ef - ED) / (k * T) - np.log(gD))


def NAm(Ef, EA, T, NA, gA=4):
    return NA * expit((EA - Ef) / (k * T) - np.log(gA))


# ============================================================
# Charge neutrality
# ============================================================
def deltaQ(Ef, T, params):
    Ec, Ev, ED, EA, Nc, Nv, ND, NA = params
    n = Ne(Ef, Ec, T, Nc)
    p = Nh(Ef, Ev, T, Nv)
    return p - n + NDp(Ef, ED, T, ND) - NAm(Ef, EA, T, NA)


# ============================================================
# Initialization
# ============================================================
def initialize():
    parser = argparse.ArgumentParser()
    parser.add_argument("--Eg", type=float, default=1.12, help="Band gap [eV]")
    parser.add_argument("--me", type=float, default=1.08, help="Electron effective mass")
    parser.add_argument("--mh", type=float, default=0.56, help="Hole effective mass")
    parser.add_argument("--ND", type=float, default=1e16, help="Donor density [cm^-3]")
    parser.add_argument("--NA", type=float, default=0.0, help="Acceptor density [cm^-3]")
    parser.add_argument("--ED", type=float, default=0.045, help="Donor level below Ec [eV]")
    parser.add_argument("--EA", type=float, default=0.045, help="Acceptor level above Ev [eV]")
    parser.add_argument("--Tmin", type=float, default=50)
    parser.add_argument("--Tmax", type=float, default=600)
    parser.add_argument("--NT", type=int, default=100)
    parser.add_argument("--Nmin", type=float, default=1e10)
    parser.add_argument("--Nmax", type=float, default=1e22)
    return parser.parse_args()


# ============================================================
# Main
# ============================================================
def main():
    args = initialize()

    Tlist = np.linspace(args.Tmin, args.Tmax, args.NT)

    Ef_list, n_list, p_list, NDp_list, NAm_list = [], [], [], [], []

    for T in Tlist:
        Ec = 0.0
        Ev = -args.Eg * e
        ED = Ec - args.ED * e
        EA = Ev + args.EA * e

        Nc = m2Nc(args.me, T)
        Nv = m2Nv(args.mh, T)

        ND = args.ND * 1e6
        NA = args.NA * 1e6

        params = (Ec, Ev, ED, EA, Nc, Nv, ND, NA)

        # --- Automatic bracket search
        Ef_min, Ef_max = Ev - 1.0 * e, Ec + 1.0 * e
        for _ in range(20):
            f1 = deltaQ(Ef_min, T, params)
            f2 = deltaQ(Ef_max, T, params)
            if f1 * f2 < 0:
                break
            Ef_min -= 0.5 * e
            Ef_max += 0.5 * e
        Ef = brentq(deltaQ, Ef_min, Ef_max, args=(T, params))

        n = Ne(Ef, Ec, T, Nc)
        p = Nh(Ef, Ev, T, Nv)

        Ef_list.append(Ef / e)
        n_list.append(n * 1e-6)
        p_list.append(p * 1e-6)
        NDp_list.append(NDp(Ef, ED, T, ND) * 1e-6)
        NAm_list.append(NAm(Ef, EA, T, NA) * 1e-6)

        print(f"T={T:6.1f} K  Ef={Ef/e: .4f} eV  n={n*1e-6:.3e}  p={p*1e-6:.3e}")

    # ========================================================
    # Plot
    # ========================================================
    fig, ax1 = plt.subplots()

    ax1.set_yscale("log")
    ax1.set_ylim(args.Nmin, args.Nmax)
    ax1.plot(Tlist, n_list, label="n")
    ax1.plot(Tlist, p_list, label="p")
    ax1.plot(Tlist, NDp_list, "--", label="ND+")
    ax1.plot(Tlist, NAm_list, "--", label="NA-")
    ax1.set_xlabel("Temperature [K]")
    ax1.set_ylabel("Density [cm$^{-3}$]")
    ax1.legend(loc="upper left")

    ax2 = ax1.twinx()
    ax2.plot(Tlist, Ef_list, "k", label="Ef")
    ax2.set_ylabel("Ef [eV]")
    ax2.legend(loc="lower right")

    plt.tight_layout()
    plt.show()


if __name__ == "__main__":
    main()
