"""
prompt
移動度フィッティングプログラムの最終版は下記のようになっています。
このコードにフィッティングパラメータを選択・固定するオプションを追加してください
"""

import pandas as pd
import numpy as np
import matplotlib.pyplot as plt
from scipy.optimize import minimize
import argparse
import os
import json

# 定数: ボルツマン定数 (eV/K)
K_B = 8.617333262e-5

# ---------------------------------------------------------
# 1. データ入出力関連の関数
# ---------------------------------------------------------
def load_hall_data(file_path):
    if not os.path.exists(file_path):
        print(f"エラー: ファイル '{file_path}' が見つかりません。")
        return None
    if file_path.endswith('.csv'):
        return pd.read_csv(file_path)
    return pd.read_excel(file_path)

def save_params(params, filename='fit_params.json'):
    with open(filename, 'w') as f:
        json.dump(params, f, indent=4)
    print(f"パラメータを保存しました: {filename}")

def load_params(filename='fit_params.json'):
    if not os.path.exists(filename):
        alt_filename = 'llsq_params.json'
        if os.path.exists(alt_filename):
            filename = alt_filename
        else:
            print(f"警告: パラメータファイルが見つかりません。デフォルト値を使用します。")
            return {'aop': 1e-3, 'a1': 1e-7, 'a2': 1e-4, 'a3': 1e1, 'VB': 0.0}
    with open(filename, 'r') as f:
        return json.load(f)

# ---------------------------------------------------------
# 2. 物理モデルと解析関連の関数
# ---------------------------------------------------------
def get_inv_mu_components(T, params, Eop):
    """
    params: [aop, a1, a2, a3, VB]
    """
    aop, a1, a2, a3, VB = params
    
    f_op = 1.0 / (np.exp(Eop / (K_B * T)) - 1.0)
    f_ac = T**1.5
    f_ni = np.ones_like(T)
    f_ii = T**-1.5
    
    components = {
        'Optical Phonon': aop * f_op,
        'Acoustic Phonon': a1 * f_ac,
        'Neutral Impurity': a2 * f_ni,
        'Ionized Impurity': a3 * f_ii
    }
    
    inv_mu_bulk = np.maximum(sum(components.values()), 1e-10)
    exp_factor = np.exp(VB / (K_B * T))
    inv_mu_total = inv_mu_bulk * exp_factor
    components['Grain Boundary'] = inv_mu_total - inv_mu_bulk
    
    return components, inv_mu_total

def solve_llsq(T, mu_exp, Eop):
    f_op = 1.0 / (np.exp(Eop / (K_B * T)) - 1.0)
    f_ac = T**1.5
    f_ni = np.ones_like(T)
    f_ii = T**-1.5
    X = np.column_stack([f_op, f_ac, f_ni, f_ii])
    y = 1.0 / mu_exp
    coeffs, _, _, _ = np.linalg.lstsq(X, y, rcond=None)
    # 負の値にならないようクリップ
    return np.maximum(coeffs, 0)

# ---------------------------------------------------------
# 3. 可視化関連の関数
# ---------------------------------------------------------
def visualize_fit(T, mu_exp, mu_fit=None, title='Hall Mobility Fit', save_name='plot.png'):
    plt.figure(figsize=(8, 6))
    plt.scatter(T, mu_exp, color='red', label='Experimental', alpha=0.6)
    if mu_fit is not None:
        idx = np.argsort(T)
        plt.plot(T[idx], mu_fit[idx], color='blue', label='Model Fit', linewidth=2)
    plt.xlabel('Temperature (K)')
    plt.ylabel('Mobility (cm²/Vs)')
#    plt.yscale('log')
    plt.title(title)
    plt.legend()
    plt.grid(True, which='both', alpha=0.3)
    plt.tight_layout()
    plt.savefig(save_name)
    plt.show()

def visualize_weights(T, components, total, save_name='weight_plot.png'):
    plt.figure(figsize=(10, 6))
    idx = np.argsort(T)
    T_sorted = T[idx]
    for name, inv_mu in components.items():
        weight = (inv_mu / total) * 100
        plt.plot(T_sorted, weight[idx], marker='o', markersize=4, label=name, linestyle='-', alpha=0.8)
    plt.xlabel('Temperature (K)')
    plt.ylabel('Contribution to Scattering (%)')
    plt.title('Scattering Mechanism Contributions')
    plt.ylim(-5, 105)
    plt.grid(True, linestyle='--', alpha=0.5)
    plt.legend(loc='best')
    plt.tight_layout()
    plt.savefig(save_name)
    plt.show()

# ---------------------------------------------------------
# 4. メイン処理
# ---------------------------------------------------------
def main():
    parser = argparse.ArgumentParser(description='Hall効果移動度フィッティング（パラメータ固定機能付き）')
    parser.add_argument('--input', type=str, default='Hall-T1.xlsx', help='入力ファイル名')
    parser.add_argument('--temp_col', type=int, default=0, help='温度列(0開始)')
    parser.add_argument('--mu_col', type=int, default=2, help='移動度列(0開始)')
    parser.add_argument('--mode', type=str, choices=['read', 'llsq', 'fit', 'weight'], default='read')
    parser.add_argument('--method', type=str, default='Nelder-Mead')
    parser.add_argument('--eop', type=float, default=0.045, help='光学フォノンエネルギー(eV)')
    parser.add_argument('--fix', type=str, default='', help='固定するパラメータをカンマ区切りで指定 (例: a2,VB)')
    args = parser.parse_args()
    
    print()
    print(f"input file={args.input}")
    print(f"mode={args.mode}")
    print(f"method={args.method}")
    print(f"eop={args.eop} eV")
    print(f"fix={args.fix}")

    df = load_hall_data(args.input)
    if df is None: return

    T = df.iloc[:, args.temp_col].values
    mu_exp = df.iloc[:, args.mu_col].values
    print()
    print("T  mu")
    for _T, _mu in zip(T, mu_exp):
        print(f"{_T:.3f}  {_mu:.4g}")

    labels = ['aop', 'a1', 'a2', 'a3', 'VB']
    fixed_list = [s.strip() for s in args.fix.split(',')] if args.fix else []

    if args.mode == 'read':
        print("\n--- 読み込みデータ ---")
        print(df)
        visualize_fit(T, mu_exp, title='Experimental Data')

    elif args.mode == 'llsq':
        c = solve_llsq(T, mu_exp, args.eop)
        params_dict = {'aop': c[0], 'a1': c[1], 'a2': c[2], 'a3': c[3], 'VB': 0.0}
        save_params(params_dict, filename='llsq_params.json')
        _, inv_fit = get_inv_mu_components(T, list(c) + [0.0], args.eop)
        visualize_fit(T, mu_exp, 1/inv_fit, title='LLSQ Initial Fit (VB=0)')

    elif args.mode == 'fit':
        p_base = load_params()
        # 初期値リストの作成
        init_full = [p_base.get(l, 0.0) for l in labels]
        
        # 最適化するインデックスと固定するインデックスを分ける
        optimize_indices = [i for i, l in enumerate(labels) if l not in fixed_list]
        print(f"最適化対象: {[labels[i] for i in optimize_indices]}")
        print(f"固定パラメータ: {fixed_list}")
        print("--- 初期パラメータ ---")
        for _label, _val in zip(labels, init_full):
            state = "(FIXED)" if _label in fixed_list else ""
            print(f"{_label:4s}: {_val:.6g} {state}")

        def objective(p_opt):
            # p_opt (動く変数) を フルリストに埋め戻す
            p_current = list(init_full)
            for idx, val in zip(optimize_indices, p_opt):
                p_current[idx] = val
            
            _, inv_total = get_inv_mu_components(T, p_current, args.eop)
            mu_model = 1.0 / inv_total
            # 移動度は正である必要があるため、パラメータが負に振れた場合のペナルティ
            if any(np.array(p_current)[:-1] < 0): return 1e18 
            
            return np.sum((np.log10(mu_exp) - np.log10(mu_model))**2) # 対数スケールでのフィッティングが安定しやすい

        init_opt = [init_full[i] for i in optimize_indices]
        res = minimize(objective, init_opt, method=args.method, options={'maxiter': 5000})
        
        # 結果の統合
        final_values = list(init_full)
        for idx, val in zip(optimize_indices, res.x):
            final_values[idx] = val
        
        final_params = {l: v for l, v in zip(labels, final_values)}
        save_params(final_params, filename='fit_params.json')
        
        print("\n--- 最適化結果 ---")
        for k, v in final_params.items():
            state = "(FIXED)" if k in fixed_list else ""
            print(f"{k:4s}: {v:.4g} {state}")
        
        _, inv_fit = get_inv_mu_components(T, final_values, args.eop)
        visualize_fit(T, mu_exp, 1/inv_fit, title='Final Fit', save_name='mu_vs_T_fit.png')

    elif args.mode == 'weight':
        p_dict = load_params()
        p_list = [p_dict.get(l, 0.0) for l in labels]
        comp, total = get_inv_mu_components(T, p_list, args.eop)
        visualize_weights(T, comp, total, save_name='mu_weight_plot.png')

if __name__ == '__main__':
    main()
