"""
Q: 等高線図ではなく、横軸に温度、縦軸に各散乱機構の寄与率を散布図＋折れ線でプロットしてください
"""

import pandas as pd
import numpy as np
import matplotlib.pyplot as plt
from scipy.optimize import minimize
import argparse
import os
import json

# 定数: ボルツマン定数 (eV/K)
K_B = 8.617333262e-5

# ---------------------------------------------------------
# 1. データ入出力関連の関数
# ---------------------------------------------------------
def load_hall_data(file_path):
    """ファイルの存在確認を行い、データを読み込む"""
    if not os.path.exists(file_path):
        print(f"エラー: ファイル '{file_path}' が見つかりません。")
        return None
    if file_path.endswith('.csv'):
        return pd.read_csv(file_path)
    return pd.read_excel(file_path)

def save_params(params, filename='fit_params.json'):
    """解析結果のパラメータをJSONで保存"""
    with open(filename, 'w') as f:
        json.dump(params, f, indent=4)
    print(f"パラメータを保存しました: {filename}")

def load_params(filename='fit_params.json'):
    """保存されたパラメータを読み込む"""
    if not os.path.exists(filename):
        # fit_paramsがない場合はllsq_paramsを探す
        alt_filename = 'llsq_params.json'
        if os.path.exists(alt_filename):
            filename = alt_filename
        else:
            print(f"エラー: パラメータファイルが見つかりません。先に mode=llsq または fit を実行してください。")
            return None
    with open(filename, 'r') as f:
        return json.load(f)

# ---------------------------------------------------------
# 2. 物理モデルと解析関連の関数
# ---------------------------------------------------------
def get_inv_mu_components(T, params, Eop):
    """各散乱機構の逆移動度 (1/mu) を個別に計算する"""
    aop, a1, a2, a3, VB = params
    
    # 各散乱機構の基底関数
    f_op = 1.0 / (np.exp(Eop / (K_B * T)) - 1.0) # 光学フォノン
    f_ac = T**1.5                                # 音響フォノン
    f_ni = np.ones_like(T)                       # 中性不純物
    f_ii = T**-1.5                               # イオン化不純物
    
    # 各成分の散乱頻度 (1/mu)
    components = {
        'Optical Phonon': aop * f_op,
        'Acoustic Phonon': a1 * f_ac,
        'Neutral Impurity': a2 * f_ni,
        'Ionized Impurity': a3 * f_ii
    }
    
    # 粒界散乱の考慮: mu_total = mu_bulk * exp(-VB/kBT)
    # => 1/mu_total = (1/mu_bulk) * exp(VB/kBT)
    inv_mu_bulk = np.maximum(sum(components.values()), 1e-10)
    exp_factor = np.exp(VB / (K_B * T))
    inv_mu_total = inv_mu_bulk * exp_factor
    
    # 粒界による散乱頻度の増加分を「粒界散乱成分」として抽出
    components['Grain Boundary'] = inv_mu_total - inv_mu_bulk
    
    return components, inv_mu_total

def solve_llsq(T, mu_exp, Eop):
    """線形最小二乗法で aop, a1, a2, a3 を推定 (VB=0)"""
    f_op = 1.0 / (np.exp(Eop / (K_B * T)) - 1.0)
    f_ac = T**1.5
    f_ni = np.ones_like(T)
    f_ii = T**-1.5
    X = np.column_stack([f_op, f_ac, f_ni, f_ii])
    y = 1.0 / mu_exp
    coeffs, _, _, _ = np.linalg.lstsq(X, y, rcond=None)
    return coeffs

# ---------------------------------------------------------
# 3. 可視化関連の関数
# ---------------------------------------------------------
def visualize_fit(T, mu_exp, mu_fit=None, title='Hall Mobility Fit', save_name='plot.png'):
    """実験データとフィッティング曲線の比較プロット"""
    plt.figure(figsize=(8, 6))
    plt.scatter(T, mu_exp, color='red', label='Experimental', alpha=0.6)
    if mu_fit is not None:
        idx = np.argsort(T)
        plt.plot(T[idx], mu_fit[idx], color='blue', label='Model Fit', linewidth=2)
    plt.xlabel('Temperature (K)')
    plt.ylabel('Mobility (cm²/Vs)')
    plt.yscale('log')
    plt.title(title)
    plt.legend()
    plt.grid(True, which='both', alpha=0.3)
    plt.tight_layout()
    plt.savefig(save_name)
    plt.show()

def visualize_weights(T, components, total, save_name='weight_plot.png'):
    """各散乱機構の寄与率を散布図＋折れ線グラフで表示"""
    plt.figure(figsize=(10, 6))
    
    # 温度でソート（折れ線を正しく引くため）
    idx = np.argsort(T)
    T_sorted = T[idx]
    
    # 各成分をループしてプロット
    for name, inv_mu in components.items():
        # 寄与率(%)の計算
        weight = (inv_mu / total) * 100
        plt.plot(T_sorted, weight[idx], marker='o', markersize=4, label=name, linestyle='-', alpha=0.8)
    
    plt.xlabel('Temperature (K)')
    plt.ylabel('Contribution to Scattering (%)')
    plt.title('Scattering Mechanism Contributions')
    plt.ylim(-5, 105) # マージンを持たせる
    plt.grid(True, linestyle='--', alpha=0.5)
    plt.legend(loc='best')
    plt.tight_layout()
    
    plt.savefig(save_name)
    print(f"寄与率グラフを保存しました: {save_name}")
    plt.show()

# ---------------------------------------------------------
# 4. メイン処理
# ---------------------------------------------------------
def main():
    parser = argparse.ArgumentParser(description='Hall効果解析 統合チュートリアルツール')
    parser.add_argument('--input', type=str, default='Hall-T1.xlsx', help='入力ファイル名')
    parser.add_argument('--temp_col', type=int, default=0, help='温度列(0開始)')
    parser.add_argument('--mu_col', type=int, default=2, help='移動度列(0開始)')
    parser.add_argument('--mode', type=str, choices=['read', 'llsq', 'fit', 'weight'], default='read')
    parser.add_argument('--eop', type=float, default=0.045, help='光学フォノンエネルギー(eV)')
    args = parser.parse_args()

    # データの読み込み
    df = load_hall_data(args.input)
    if df is None: return

    T = df.iloc[:, args.temp_col].values
    mu_exp = df.iloc[:, args.mu_col].values

    if args.mode == 'read':
        print("\n--- 読み込みデータ ---")
        print(df)
        visualize_fit(T, mu_exp, title='Experimental Data')

    elif args.mode == 'llsq':
        c = solve_llsq(T, mu_exp, args.eop)
        params_dict = {'aop': c[0], 'a1': c[1], 'a2': c[2], 'a3': c[3], 'VB': 0.0}
        save_params(params_dict, filename='llsq_params.json')
        
        _, inv_fit = get_inv_mu_components(T, list(c) + [0.0], args.eop)
        visualize_fit(T, mu_exp, 1/inv_fit, title='LLSQ Initial Fit (VB=0)')

    elif args.mode == 'fit':
        p_base = load_params() # llsqの結果を初期値として読み込む
        if p_base is None: return
        init = [p_base['aop'], p_base['a1'], p_base['a2'], p_base['a3'], 0.0]
        
        def objective(p):
            _, inv_total = get_inv_mu_components(T, p, args.eop)
            mu_model = 1.0 / inv_total
            # 残差平方和の最小化
            return np.sum((mu_exp - mu_model)**2)
        
        print("非線形最適化中 (Nelder-Mead)...")
        res = minimize(objective, init, method='Nelder-Mead', options={'maxiter': 2000})
        
        labels = ['aop', 'a1', 'a2', 'a3', 'VB']
        final_params = {l: v for l, v in zip(labels, res.x)}
        save_params(final_params, filename='fit_params.json')
        
        print("\n--- 最適化後のパラメータ ---")
        for k, v in final_params.items():
            print(f"{k:4s}: {v:.4e}")
        
        _, inv_fit = get_inv_mu_components(T, res.x, args.eop)
        visualize_fit(T, mu_exp, 1/inv_fit, title=f'Final Fit (VB={res.x[4]:.4e} eV)', save_name='mu_vs_T_fit.png')

    elif args.mode == 'weight':
        # 保存されたパラメータを読み込む
        p_dict = load_params()
        if p_dict is None: return
        p_list = [p_dict['aop'], p_dict['a1'], p_dict['a2'], p_dict['a3'], p_dict.get('VB', 0.0)]
        
        # 各成分の計算とプロット
        comp, total = get_inv_mu_components(T, p_list, args.eop)
        visualize_weights(T, comp, total, save_name='mu_weight_plot.png')

if __name__ == '__main__':
    main()