"""
prompt
粒界のポテンシャル障壁 VBによる散乱を入れてフィッティングしてください
# mode=fitで非線形最小二乗法を実行する。アルゴリズムはnelder-meadを使う
# 移動度モデルは exp(-VB / kBT) * (光学フォノン散乱などから計算した移動度)
# VBの初期値は0、光学フォノンなどの散乱パラメータは、mode=llsqの結果をファイルに保存したものを読み込む
"""

import pandas as pd
import numpy as np
import matplotlib.pyplot as plt
from scipy.optimize import minimize
import argparse
import os
import json

# 定数: ボルツマン定数 (eV/K)
K_B = 8.617333262e-5

def load_hall_data(file_path):
    if not os.path.exists(file_path):
        print(f"エラー: ファイル '{file_path}' が見つかりません。")
        return None
    return pd.read_csv(file_path) if file_path.endswith('.csv') else pd.read_excel(file_path)

def save_params(params, filename='llsq_params.json'):
    with open(filename, 'w') as f:
        json.dump(params, f, indent=4)
    print(f"パラメータを保存しました: {filename}")

def load_params(filename='llsq_params.json'):
    if not os.path.exists(filename):
        print(f"エラー: パラメータファイル '{filename}' が見つかりません。先に mode=llsq を実行してください。")
        return None
    with open(filename, 'r') as f:
        return json.load(f)

def mobility_model(T, params, Eop):
    """
    移動度モデル: mu = mu_bulk * exp(-VB / kBT)
    params: [aop, a1, a2, a3, VB]
    """
    aop, a1, a2, a3, VB = params
    
    # 各散乱の基底関数
    f_op = 1.0 / (np.exp(Eop / (K_B * T)) - 1.0)
    f_ac = T**1.5
    f_ni = np.ones_like(T)
    f_ii = T**-1.5
    
    # 1/mu_bulk の計算
    inv_mu_bulk = aop * f_op + a1 * f_ac + a2 * f_ni + a3 * f_ii
    # ゼロ除算防止
    inv_mu_bulk = np.maximum(inv_mu_bulk, 1e-10)
    mu_bulk = 1.0 / inv_mu_bulk
    
    # 粒界ポテンシャル障壁の効果を乗算
    return mu_bulk * np.exp(-VB / (K_B * T))

def solve_llsq(T, mu_exp, Eop):
    """線形最小二乗法で aop, a1, a2, a3 を推定"""
    f_op = 1.0 / (np.exp(Eop / (K_B * T)) - 1.0)
    f_ac = T**1.5
    f_ni = np.ones_like(T)
    f_ii = T**-1.5
    X = np.column_stack([f_op, f_ac, f_ni, f_ii])
    y = 1.0 / mu_exp
    coeffs, _, _, _ = np.linalg.lstsq(X, y, rcond=None)
    return coeffs

def visualize(T, mu_exp, mu_calc=None, title='Hall Plot', save_name='plot.png'):
    plt.figure(figsize=(10, 6))
    plt.scatter(T, mu_exp, color='red', label='Experimental', zorder=5)
    if mu_calc is not None:
        idx = np.argsort(T)
        plt.plot(T[idx], mu_calc[idx], color='blue', label='Model Fit', linewidth=2)
    plt.xlabel('Temperature (K)')
    plt.ylabel('Mobility (cm²/Vs)')
    plt.title(title)
    plt.legend()
    plt.grid(True, alpha=0.3)
    plt.tight_layout()
    plt.savefig(save_name)
    print(f"グラフを保存しました: {save_name}")
    plt.show()

def main():
    parser = argparse.ArgumentParser(description='Hall効果解析: 非線形最小二乗法 (Nelder-Mead)')
    parser.add_argument('--input', type=str, default='Hall-T1.xlsx', help='入力ファイル')
    parser.add_argument('--temp_col', type=int, default=0, help='温度列')
    parser.add_argument('--mu_col', type=int, default=2, help='移動度列')
    parser.add_argument('--mode', type=str, choices=['read', 'llsq', 'fit'], default='read')
    parser.add_argument('--eop', type=float, default=0.045, help='Eop (eV)')
    args = parser.parse_args()

    df = load_hall_data(args.input)
    if df is None: return
    
    print("\n--- [Data Preview] ---")
    print(df.head())
    
    T = df.iloc[:, args.temp_col].values
    mu_exp = df.iloc[:, args.mu_col].values

    if args.mode == 'read':
        visualize(T, mu_exp, title='Experimental Data')

    elif args.mode == 'llsq':
        coeffs = solve_llsq(T, mu_exp, args.eop)
        params_dict = {'aop': coeffs[0], 'a1': coeffs[1], 'a2': coeffs[2], 'a3': coeffs[3]}
        save_params(params_dict)
        
        # 簡易プロット用
        mu_llsq = mobility_model(T, list(coeffs) + [0.0], args.eop)
        visualize(T, mu_exp, mu_llsq, title='LLSQ Initial Fit (VB=0)')

    elif args.mode == 'fit':
        p_base = load_params()
        if p_base is None: return
        
        # 初期値設定 [aop, a1, a2, a3, VB]
        initial_guess = [p_base['aop'], p_base['a1'], p_base['a2'], p_base['a3'], 0.0]
        
        # 目的関数: 残差平方和
        def cost_func(p):
            mu_model = mobility_model(T, p, args.eop)
            return np.sum((mu_exp - mu_model)**2)
        
        print("最適化中 (Nelder-Mead)...")
        res = minimize(cost_func, initial_guess, method='Nelder-Mead')
        
        final_p = res.x
        print("\n--- [最適化結果] ---")
        labels = ['aop', 'a1', 'a2', 'a3', 'VB']
        for l, v in zip(labels, final_p):
            print(f"{l:4s}: {v:.4e}")
        
        mu_fit = mobility_model(T, final_p, args.eop)
        visualize(T, mu_exp, mu_fit, title=f'Non-linear Fit (VB={final_p[4]:.4e} eV)', save_name='mu_vs_T_fit.png')

if __name__ == '__main__':
    main()