"""
prompt
次の機能追加をしてください
# 読み込んだデータをコンソールに出力してください
# main()関数が大きすぎます。役割別に関数化してください
# 大きなtryブロックは使わないでください。ファイルの存在確認など、事前に想定できる例外処理はifをつかって処理してください
# mode=llsqを実装してください
- 散乱機構として、光学フォノン散乱 (tau^-1 = aop / (exp(Eop/kBT) - 1)、音響フォノン散乱 a1*T^1.5, 中性不純物散乱 a2*T^0、イオン化不純物散乱 a3*T^-1.5 を考慮する
- Eopはデフォルト 0.045 eVの定数パラメータとする
- 線形最小二乗法で aop, a1, a2, a3を求める
- aop, a1, a2, a3をコンソールに出力する
- 最適化パラメータを使って μ(T) を計算し、入力データと比較して可視化する
- グラフはファイルに保存するとともに、matplotlibでウインドウに描画する
"""

import pandas as pd
import numpy as np
import matplotlib.pyplot as plt
import argparse
import os

# 定数: ボルツマン定数 (eV/K)
K_B = 8.617333262e-5

def load_hall_data(file_path):
    """ファイルの存在確認を行い、データを読み込む"""
    if not os.path.exists(file_path):
        print(f"エラー: ファイル '{file_path}' が見つかりません。")
        return None
    
    # 拡張子による読み込み分岐
    if file_path.endswith('.csv'):
        return pd.read_csv(file_path)
    else:
        return pd.read_excel(file_path)

def solve_mobility_llsq(T, mu_exp, eop_val):
    """線形最小二乗法を用いて各散乱係数を算出する"""
    # 1/mu = a_op*f_op + a1*T^1.5 + a2 + a3*T^-1.5
    # 基底関数の定義
    f_op = 1.0 / (np.exp(eop_val / (K_B * T)) - 1.0)
    f_ac = T**1.5
    f_ni = np.ones_like(T)
    f_ii = T**-1.5
    
    # デザイン行列の作成 [N x 4]
    X = np.column_stack([f_op, f_ac, f_ni, f_ii])
    y = 1.0 / mu_exp
    
    # 最小二乗法を実行: a = (X^T X)^-1 X^T y
    coeffs, residuals, rank, s = np.linalg.lstsq(X, y, rcond=None)
    
    # フィッティング結果の mu を再計算
    inv_mu_fit = X @ coeffs
    mu_fit = 1.0 / inv_mu_fit
    
    return coeffs, mu_fit

def visualize_results(T, mu_exp, mu_fit=None, labels=None, save_name='plot.png'):
    """データのプロットと保存、表示を行う"""
    plt.figure(figsize=(10, 6))
    plt.scatter(T, mu_exp, color='red', label='Experimental Data', zorder=5)
    
    if mu_fit is not None:
        # 曲線を描くために温度順にソート
        sort_idx = np.argsort(T)
        plt.plot(T[sort_idx], mu_fit[sort_idx], color='blue', 
                 label='LLSQ Fit (Total)', linewidth=2)
    
    plt.xlabel(labels[0] if labels else 'Temperature (K)')
    plt.ylabel(labels[1] if labels else r'Mobility ($\mathrm{cm^2/Vs}$)')
    plt.title('Hall Effect Analysis: Mobility vs Temperature')
    plt.legend()
    plt.grid(True, which='both', linestyle='--', alpha=0.5)
    plt.tight_layout()
    
    # ファイル保存
    plt.savefig(save_name)
    print(f"グラフを保存しました: {save_name}")
    
    # ウインドウ表示 (ローカル環境用)
    plt.show()

def main():
    parser = argparse.ArgumentParser(description='Hall効果データの高度な解析チュートリアル')
    parser.add_argument('--input', type=str, default='Hall-T1.xlsx', help='入力ファイル名')
    parser.add_argument('--temp_col', type=int, default=0, help='温度の列番号')
    parser.add_argument('--mu_col', type=int, default=2, help='移動度の列番号 (提供データでは 2)')
    parser.add_argument('--mode', type=str, choices=['read', 'llsq'], default='read', help='実行モード')
    parser.add_argument('--eop', type=float, default=0.045, help='光学フォノンエネルギー (eV)')
    
    args = parser.parse_args()

    # 1. データの読み込み
    df = load_hall_data(args.input)
    if df is None:
        return

    # 2. データの出力
    print("\n--- [データ確認] ---")
    print(df)
    print("--------------------\n")

    # データの抽出
    T = df.iloc[:, args.temp_col].values
    mu = df.iloc[:, args.mu_col].values
    labels = [df.columns[args.temp_col], df.columns[args.mu_col]]

    # 3. モード別処理
    if args.mode == 'read':
        visualize_results(T, mu, labels=labels, save_name='mu_vs_T_read.png')
        
    elif args.mode == 'llsq':
        # 最小二乗法計算
        coeffs, mu_fit = solve_mobility_llsq(T, mu, args.eop)
        aop, a1, a2, a3 = coeffs
        
        # 結果の出力
        print("--- [LLSQ フィッティング結果] ---")
        print(f"光学フォノン係数 (aop): {aop:.4e}")
        print(f"音響フォノン係数 (a1) : {a1:.4e}")
        print(f"中性不純物係数   (a2) : {a2:.4e}")
        print(f"イオン化不純物係数(a3) : {a3:.4e}")
        print("---------------------------------\n")
        
        # 可視化
        visualize_results(T, mu, mu_fit=mu_fit, labels=labels, save_name='mu_vs_T_llsq.png')

if __name__ == '__main__':
    main()