"""
prompt
Hall効果測定で温度T, 移動度mu, キャリア濃度Neを測定した、添付のExcelファイルがあります。まず、以下の仕様で、可視化するプログラムを作ってください

# argparseを使い、入力Excelファイル名 (デフォルト Hall-T1.xlsx)、温度の入っている列番号（デフォルト 0)、移動度の入っている列番号 (デフォルト 1)を読み込みます。

# modeパラメータを用意し、mode=readの場合、データファイルを読み込み、mu(T)グラフをプロットしてください
"""

import pandas as pd
import matplotlib.pyplot as plt
import argparse
import os

def main():
    # 1. コマンドライン引数の設定
    parser = argparse.ArgumentParser(description='Hall効果測定データの可視化チュートリアル')
    
    # 入力ファイル名 (デフォルト: Hall-T1.xlsx)
    parser.add_argument('--input', type=str, default='Hall-T1.xlsx', 
                        help='入力Excelファイル名 (デフォルト: Hall-T1.xlsx)')
    
    # 温度の列番号 (デフォルト: 0)
    parser.add_argument('--temp_col', type=int, default=0, 
                        help='温度(T)が入っている列番号 (0開始, デフォルト: 0)')
    
    # 移動度の列番号 (デフォルト: 1)
    parser.add_argument('--mu_col', type=int, default=1, 
                        help='移動度(mu)が入っている列番号 (0開始, デフォルト: 1)')
    
    # 実行モード (現在は 'read' のみ)
    parser.add_argument('--mode', type=str, choices=['read'], default='read', 
                        help='実行モード (read: データの読み込みとプロット)')

    args = parser.parse_args()

    # 2. mode=read の場合の処理
    if args.mode == 'read':
        print(f"ファイルを読み込んでいます: {args.input}")
        
        try:
            # ファイルの拡張子に応じて読み込み方法を分岐
            if args.input.endswith('.csv'):
                df = pd.read_csv(args.input)
            else:
                # Excelファイルを読み込む場合は openpyxl 等が必要
                df = pd.read_excel(args.input)
            
            # 指定された列からデータを抽出 (ilocを使用)
            # args.temp_col, args.mu_col は0から始まるインデックス
            t_data = df.iloc[:, args.temp_col]
            mu_data = df.iloc[:, args.mu_col]
            
            # グラフのラベル用に列名を取得
            t_label = df.columns[args.temp_col]
            mu_label = df.columns[args.mu_col]

            # 3. グラフのプロット
            plt.figure(figsize=(8, 6))
            plt.plot(t_data, mu_data, marker='o', linestyle='-', color='b', label=mu_label)
            
            plt.xlabel(t_label)
            plt.ylabel(mu_label)
            plt.title(f'Hall Effect Measurement: {mu_label} vs {t_label}')
            plt.grid(True, which='both', linestyle='--', alpha=0.5)
            plt.legend()
            
            # 画像として保存
            output_fname = 'mu_vs_T_plot.png'
            plt.savefig(output_fname)
            print(f"グラフを保存しました: {output_fname}")
            plt.show()
            
        except Exception as e:
            print(f"エラーが発生しました: {e}")

if __name__ == '__main__':
    main()
    