"""
#Prompt #all
半導体のキャリア統計をシミュレートする、以下の仕様を満たすPythonプログラムを作成してください。

1. 【物理モデル】
- 伝導帯、価電子帯、ドナー/アクセプター準位を考慮し、電荷中性条件 (p - n + ND+ - NA- = 0) を満たすフェルミ準位 Ef を決定する。
- キャリア密度 (n, p) は、ボルツマン近似ではなく「フェルミ・ディラック積分 (オーダー1/2)」を用いて計算すること。
- 不純物のイオン化 (ND+, NA-) は、占有関数と退化因子 (gD=2, gA=4) を用いて計算すること。

2. 【数値計算の安定化】
- 強い縮退（Efがバンド内にある状態）でも計算できるよう、指数関数のオーバーフローを避けるために `scipy.special.expit` を使用した数値積分を行うこと。
- Efの決定には `scipy.optimize.brentq` (Brent法) を使用し、解を囲む範囲 (bracket) が不明な場合は自動で符号反転を確認するまで探索範囲を広げるロジックを実装すること。

3. 【プログラム構造】
- `argparse` を使用して、バンドギャップ(Eg)、有効質量(me, mh)、不純物密度(ND, NA)などのパラメータを外部から与えられるようにする。
- `initialize()` 関数でパラメータの初期設定を行い、`main()` で計算とプロットを行う。
- `if __name__ == "__main__":` ギミックを使用すること。
- 各関数名（m2Nc, m2Nv, fe, fh, Fj, Ne, Nh, NDp, NAm, deltaQ）は短く、物理的な意味を反映させること。

4. 【出力とプロット】
- 指定された温度範囲（例: 50K〜600K）でスイープを行い、結果をコンソールに表示する。
- グラフ表示の密度範囲は、Nmin,Maxパラメータで指定できるようにする (デフォルト Nmin=1e10, Nmax=1e22)
- グラフは2軸プロットとし、左軸に「キャリア密度・イオン化不純物密度 (ログスケール)」、右軸に「フェルミ準位 Ef (リニアスケール)」を描画すること。
"""

#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Semiconductor carrier statistics simulator (Fermi-Dirac, strong-degeneracy safe)

Model:
  - Conduction band / valence band + donor/acceptor levels
  - Charge neutrality:  p - n + ND+ - NA- = 0  -> solve Ef(T)
  - n, p use Fermi–Dirac integral F_{1/2} (numerical integration)
  - Dopant ionization uses expit() with degeneracy factors (gD=2, gA=4)
  - Ef root finding uses brentq with automatic bracketing expansion

Conventions:
  - Energies in eV
  - Ev = 0 eV, Ec = Eg eV, Ef measured from Ev
  - Dopant densities in cm^-3, effective masses in units of m0
"""

#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Semiconductor carrier statistics simulator (Fermi-Dirac, strong-degeneracy safe)

Model:
  - Conduction band / valence band + donor/acceptor levels
  - Charge neutrality:  p - n + ND+ - NA- = 0  -> solve Ef(T)
  - n, p use Fermi–Dirac integral F_{1/2} (numerical integration)
  - Dopant ionization uses expit() with degeneracy factors (gD=2, gA=4)
  - Ef root finding uses brentq with automatic bracketing expansion

Conventions:
  - Energies in eV
  - Ev = 0 eV, Ec = Eg eV, Ef measured from Ev
  - Dopant densities in cm^-3, effective masses in units of m0
"""

import argparse
import math
import numpy as np
import matplotlib.pyplot as plt

from scipy import integrate
from scipy.optimize import brentq
from scipy.special import expit

# -------------------------
# constants (SI + eV)
# -------------------------
q = 1.602176634e-19       # C
kB_J = 1.380649e-23       # J/K
kB_eV = 8.617333262145e-5 # eV/K
h = 6.62607015e-34        # J*s
m0 = 9.1093837015e-31     # kg
pi = math.pi


# ============================================================
# density of states prefactors
# Nc, Nv in cm^-3 for given T and effective mass (in m0)
# ============================================================
def m2Nc(T, me):
    """Effective DOS in conduction band Nc(T) [cm^-3]."""
    m_eff = me * m0
    Nc_m3 = 2.0 * (2.0 * pi * m_eff * kB_J * T / (h * h)) ** 1.5
    return Nc_m3 / 1.0e6  # m^-3 -> cm^-3

def m2Nv(T, mh):
    """Effective DOS in valence band Nv(T) [cm^-3]."""
    m_eff = mh * m0
    Nv_m3 = 2.0 * (2.0 * pi * m_eff * kB_J * T / (h * h)) ** 1.5
    return Nv_m3 / 1.0e6  # m^-3 -> cm^-3


# ============================================================
# Fermi-Dirac integral F_{1/2}(eta)
#   F_{1/2}(eta) = (2/sqrt(pi)) * ∫_0^∞ sqrt(e) / (1 + exp(e - eta)) de
# Use substitution e = x^2:
#   F_{1/2}(eta) = (4/sqrt(pi)) * ∫_0^∞ x^2 / (1 + exp(x^2 - eta)) dx
# For stability use expit(eta - x^2).
# ============================================================
def Fj(eta):
    """Compute Fermi–Dirac integral F_{1/2}(eta) via stable numerical integration."""
    # integrand in x-space: (4/sqrt(pi)) * x^2 * expit(eta - x^2)
    c = 4.0 / math.sqrt(pi)

    def integrand(x):
        # expit(eta - x^2) = 1/(1+exp(x^2-eta)) stable
        return c * (x * x) * expit(eta - x * x)

    # quad over [0, inf); limit helps for sharper features in strong degeneracy
    val, _err = integrate.quad(integrand, 0.0, np.inf, epsabs=1e-12, epsrel=1e-10, limit=400)
    return val


# ============================================================
# helper "short" physics functions
# ============================================================
def fe(E, Ef, T):
    """Fermi occupation f(E) = 1/(1+exp((E-Ef)/kT)) using expit."""
    return expit((Ef - E) / (kB_eV * T))

def fh(E, Ef, T):
    """Hole occupation = 1 - f(E)."""
    return 1.0 - fe(E, Ef, T)

def Ne(Ef, T, Eg, me):
    """Electron density n [cm^-3] using Nc*F_{1/2}."""
    Ec = Eg
    eta = (Ef - Ec) / (kB_eV * T)
    return m2Nc(T, me) * Fj(eta)

def Nh(Ef, T, mh):
    """Hole density p [cm^-3] using Nv*F_{1/2}."""
    Ev = 0.0
    eta = (Ev - Ef) / (kB_eV * T)
    return m2Nv(T, mh) * Fj(eta)

def NDp(Ef, T, Eg, ED, ND, gD=2.0):
    """
    Ionized donor density ND+ [cm^-3].
    Donor level: E_D = Ec - ED (ED is depth below Ec, in eV).
    ND+ = ND / (1 + gD*exp((Ef - E_D)/kT))
        = ND * expit((E_D - Ef)/kT - ln(gD))
    """
    Ec = Eg
    E_D = Ec - ED
    x = (E_D - Ef) / (kB_eV * T) - math.log(gD)
    return ND * expit(x)

def NAm(Ef, T, EA, NA, gA=4.0):
    """
    Ionized acceptor density NA- [cm^-3].
    Acceptor level: E_A = Ev + EA (EA is height above Ev, in eV).
    NA- = NA / (1 + gA*exp((E_A - Ef)/kT))
        = NA * expit((Ef - E_A)/kT - ln(gA))
    """
    Ev = 0.0
    E_A = Ev + EA
    x = (Ef - E_A) / (kB_eV * T) - math.log(gA)
    return NA * expit(x)

def deltaQ(Ef, T, Eg, me, mh, ND, NA, ED, EA, gD=2.0, gA=4.0):
    """Charge neutrality residual: p - n + ND+ - NA-."""
    n = Ne(Ef, T, Eg, me)
    p = Nh(Ef, T, mh)
    ndp = NDp(Ef, T, Eg, ED, ND, gD=gD)
    nam = NAm(Ef, T, EA, NA, gA=gA)
    return p - n + ndp - nam


# ============================================================
# bracketing logic for brentq
#   - Start from a reasonable window around the bands
#   - Expand until sign change
#   - If still no sign change, scan and find a sign-changing pair
# ============================================================
def bracket_Ef(T, Eg, me, mh, ND, NA, ED, EA, gD=2.0, gA=4.0):
    Ev = 0.0
    Ec = Eg

    # initial window (eV) around band edges
    W0 = max(1.0, 2.0 * Eg)  # ensure non-tiny
    a = Ev - W0
    b = Ec + W0

    fa = deltaQ(a, T, Eg, me, mh, ND, NA, ED, EA, gD=gD, gA=gA)
    fb = deltaQ(b, T, Eg, me, mh, ND, NA, ED, EA, gD=gD, gA=gA)

    # Expand symmetrically
    W = W0
    for _ in range(40):
        if fa == 0.0:
            return a, a
        if fb == 0.0:
            return b, b
        if fa * fb < 0.0:
            return a, b
        W *= 2.0
        a = Ev - W
        b = Ec + W
        fa = deltaQ(a, T, Eg, me, mh, ND, NA, ED, EA, gD=gD, gA=gA)
        fb = deltaQ(b, T, Eg, me, mh, ND, NA, ED, EA, gD=gD, gA=gA)

    # Fallback: scan for sign change in a wide range
    a = Ev - W
    b = Ec + W
    xs = np.linspace(a, b, 801)
    fs = np.array([deltaQ(x, T, Eg, me, mh, ND, NA, ED, EA, gD=gD, gA=gA) for x in xs])

    # find adjacent sign change
    s = np.sign(fs)
    s[s == 0] = 1  # treat exact zeros as + for adjacency logic
    idx = np.where(s[:-1] * s[1:] < 0)[0]
    if len(idx) > 0:
        i = int(idx[0])
        return float(xs[i]), float(xs[i + 1])

    raise RuntimeError("Failed to bracket Ef: deltaQ did not change sign even after expansion+scan.")


def solve_Ef(T, Eg, me, mh, ND, NA, ED, EA, gD=2.0, gA=4.0):
    a, b = bracket_Ef(T, Eg, me, mh, ND, NA, ED, EA, gD=gD, gA=gA)
    if a == b:
        return a
    f = lambda x: deltaQ(x, T, Eg, me, mh, ND, NA, ED, EA, gD=gD, gA=gA)
    return brentq(f, a, b, xtol=1e-12, rtol=1e-10, maxiter=200)


# ============================================================
# initialization + main
# ============================================================
def initialize():
    p = argparse.ArgumentParser(description="Semiconductor carrier statistics (Fermi-Dirac, brentq, stable expit).")

    # physics params
    p.add_argument("--Eg", type=float, default=1.12, help="Bandgap Eg [eV] (Ev=0, Ec=Eg).")
    p.add_argument("--me", type=float, default=1.08, help="Electron effective mass (in m0).")
    p.add_argument("--mh", type=float, default=0.55, help="Hole effective mass (in m0).")

    p.add_argument("--ND", type=float, default=1e17, help="Donor density ND [cm^-3].")
    p.add_argument("--NA", type=float, default=1e15,  help="Acceptor density NA [cm^-3].")

    p.add_argument("--ED", type=float, default=0.045, help="Donor level depth below Ec: ED [eV] (E_D=Ec-ED).")
    p.add_argument("--EA", type=float, default=0.045, help="Acceptor level height above Ev: EA [eV] (E_A=Ev+EA).")

    p.add_argument("--gD", type=float, default=1.0, help="Donor degeneracy factor gD (default 2).")
    p.add_argument("--gA", type=float, default=1.0, help="Acceptor degeneracy factor gA (default 4).")

    # temperature sweep
    p.add_argument("--Tmin", type=float, default=50.0,  help="Min temperature [K].")
    p.add_argument("--Tmax", type=float, default=600.0, help="Max temperature [K].")
    p.add_argument("--nT",   type=int,   default=56,    help="Number of temperature points.")

    # plot controls
    p.add_argument("--Nmin", type=float, default=1e10, help="Min y (density) for log plot [cm^-3].")
    p.add_argument("--Nmax", type=float, default=1e22, help="Max y (density) for log plot [cm^-3].")

    p.add_argument("--no-plot", action="store_true", help="Do not show plot (console output only).")

    return p.parse_args()


def main():
    args = initialize()

    Ts = np.linspace(args.Tmin, args.Tmax, args.nT)

    Ef_list = []
    n_list = []
    p_list = []
    ndp_list = []
    nam_list = []

    print("# T[K]         Ef[eV]           n[cm^-3]         p[cm^-3]        ND+[cm^-3]       NA-[cm^-3]")
    for T in Ts:
        Ef = solve_Ef(T, args.Eg, args.me, args.mh, args.ND, args.NA, args.ED, args.EA, gD=args.gD, gA=args.gA)
        n  = Ne(Ef, T, args.Eg, args.me)
        p  = Nh(Ef, T, args.mh)
        ndp = NDp(Ef, T, args.Eg, args.ED, args.ND, gD=args.gD)
        nam = NAm(Ef, T, args.EA, args.NA, gA=args.gA)

        Ef_list.append(Ef)
        n_list.append(n)
        p_list.append(p)
        ndp_list.append(ndp)
        nam_list.append(nam)

        print(f"{T:7.2f}  {Ef:14.8f}  {n:14.6e}  {p:14.6e}  {ndp:14.6e}  {nam:14.6e}")

    Ef_arr = np.array(Ef_list)
    n_arr = np.array(n_list)
    p_arr = np.array(p_list)
    ndp_arr = np.array(ndp_list)
    nam_arr = np.array(nam_list)

    if args.no_plot:
        return

    fig, ax1 = plt.subplots(figsize=(6, 6))
    ax2 = ax1.twinx()

    # left axis: densities (log)
    ax1.plot(Ts, n_arr, label="n")
    ax1.plot(Ts, p_arr, label="p")
    ax1.plot(Ts, ndp_arr, label="ND+")
    ax1.plot(Ts, nam_arr, label="NA-")
    ax1.set_yscale("log")
    ax1.set_ylim(args.Nmin, args.Nmax)
    ax1.set_xlabel("Temperature T [K]")
    ax1.set_ylabel("Density [cm$^{-3}$] (log)")

    # right axis: Ef (linear)
    ax2.plot(Ts, Ef_arr, label="Ef", linestyle="--")
    ax2.set_ylabel("Fermi level Ef [eV] (linear)")

    # combined legend
    lines1, labels1 = ax1.get_legend_handles_labels()
    lines2, labels2 = ax2.get_legend_handles_labels()
    ax1.legend(lines1 + lines2, labels1 + labels2, loc="best")

    ax1.grid(True, which="both", linestyle=":")
    plt.tight_layout()
    plt.show()


if __name__ == "__main__":
    main()
