"""
#Prompt #all
半導体のキャリア統計をシミュレートする、以下の仕様を満たすPythonプログラムを作成してください。

1. 【物理モデル】
- 伝導帯、価電子帯、ドナー/アクセプター準位を考慮し、電荷中性条件 (p - n + ND+ - NA- = 0) を満たすフェルミ準位 Ef を決定する。
- キャリア密度 (n, p) は、ボルツマン近似ではなく「フェルミ・ディラック積分 (オーダー1/2)」を用いて計算すること。
- 不純物のイオン化 (ND+, NA-) は、占有関数と退化因子 (gD=2, gA=4) を用いて計算すること。

2. 【数値計算の安定化】
- 強い縮退（Efがバンド内にある状態）でも計算できるよう、指数関数のオーバーフローを避けるために `scipy.special.expit` を使用した数値積分を行うこと。
- Efの決定には `scipy.optimize.brentq` (Brent法) を使用し、解を囲む範囲 (bracket) が不明な場合は自動で符号反転を確認するまで探索範囲を広げるロジックを実装すること。

3. 【プログラム構造】
- `argparse` を使用して、バンドギャップ(Eg)、有効質量(me, mh)、不純物密度(ND, NA)などのパラメータを外部から与えられるようにする。
- `initialize()` 関数でパラメータの初期設定を行い、`main()` で計算とプロットを行う。
- `if __name__ == "__main__":` ギミックを使用すること。
- 各関数名（m2Nc, m2Nv, fe, fh, Fj, Ne, Nh, NDp, NAm, deltaQ）は短く、物理的な意味を反映させること。

4. 【出力とプロット】
- 指定された温度範囲（例: 50K〜600K）でスイープを行い、結果をコンソールに表示する。
- グラフ表示の密度範囲は、Nmin,Maxパラメータで指定できるようにする (デフォルト Nmin=1e10, Nmax=1e22)
- グラフは2軸プロットとし、左軸に「キャリア密度・イオン化不純物密度 (ログスケール)」、右軸に「フェルミ準位 Ef (リニアスケール)」を描画すること。
"""

import numpy as np
import matplotlib.pyplot as plt
from scipy import integrate, optimize, special
from scipy.constants import k, hbar, m_e, e
import argparse

# --- 物理定数の設定 ---
KbT_const = k / e  # [eV/K]

def initialize():
    parser = argparse.ArgumentParser(description="Semiconductor Carrier Statistics Simulator")
    parser.add_argument("--Eg", type=float, default=1.12, help="Bandgap [eV] (default: 1.12 for Si)")
    parser.add_argument("--me", type=float, default=1.08, help="Effective mass of electrons / m0")
    parser.add_argument("--mh", type=float, default=0.55, help="Effective mass of holes / m0")
    parser.add_argument("--ND", type=float, default=1e17, help="Donor density [cm^-3]")
    parser.add_argument("--NA", type=float, default=1e15, help="Acceptor density [cm^-3]")
    parser.add_argument("--ED", type=float, default=0.045, help="Donor ionization energy below Ec [eV]")
    parser.add_argument("--EA", type=float, default=0.045, help="Acceptor ionization energy above Ev [eV]")
    parser.add_argument("--Tmin", type=float, default=50, help="Min Temperature [K]")
    parser.add_argument("--Tmax", type=float, default=600, help="Max Temperature [K]")
    parser.add_argument("--Nmin", type=float, default=1e10, help="Plot Min Density [cm^-3]")
    parser.add_argument("--Nmax", type=float, default=1e22, help="Plot Max Density [cm^-3]")
    return parser.parse_args()

# --- 物理計算関数 ---

def m2Nc(T, me_eff):
    # 有効状態密度 Nc [cm^-3]
    val = 2 * (me_eff * m_e * k * T / (2 * np.pi * hbar**2))**(1.5)
    return val * 1e-6 # m^-3 to cm^-3

def m2Nv(T, mh_eff):
    # 有効状態密度 Nv [cm^-3]
    val = 2 * (mh_eff * m_e * k * T / (2 * np.pi * hbar**2))**(1.5)
    return val * 1e-6

def Fj(eta):
    # フェルミ・ディラック積分 (j=1/2)
    # overflow回避のため expit(x) = 1/(1+exp(-x)) を利用
    integrand = lambda x: np.sqrt(x) * special.expit(eta - x)
    res, _ = integrate.quad(integrand, 0, 500) # 十分な範囲まで積分
    return res

def Ne(Ef, Ec, T, Nc):
    eta = (Ef - Ec) / (KbT_const * T)
    return Nc * (2 / np.sqrt(np.pi)) * Fj(eta)

def Nh(Ef, Ev, T, Nv):
    eta = (Ev - Ef) / (KbT_const * T)
    return Nv * (2 / np.sqrt(np.pi)) * Fj(eta)

def NDp(Ef, Ec, ED, ND, T):
    # イオン化ドナー密度 (gD=2)
    E_donor = Ec - ED
    return ND * (1 - 1 / (1 + 0.5 * np.exp((E_donor - Ef) / (KbT_const * T))))

def NAm(Ef, Ev, EA, NA, T):
    # イオン化アクセプター密度 (gA=4)
    E_acc = Ev + EA
    return NA / (1 + 4 * np.exp((E_acc - Ef) / (KbT_const * T)))

def deltaQ(Ef, Ec, Ev, ED, EA, ND, NA, T, Nc, Nv):
    # 電荷中性条件の残差: p - n + ND+ - NA-
    n = Ne(Ef, Ec, T, Nc)
    p = Nh(Ef, Ev, T, Nv)
    ndp = NDp(Ef, Ec, ED, ND, T)
    nam = NAm(Ef, Ev, EA, NA, T)
    return p - n + ndp - nam

def main():
    args = initialize()
    T_list = np.linspace(args.Tmin, args.Tmax, 50)
    
    results = { "T": [], "Ef": [], "n": [], "p": [], "NDp": [], "NAm": [] }

    print(f"{'T [K]':>8} | {'Ef [eV]':>8} | {'n [cm^-3]':>10} | {'p [cm^-3]':>10}")
    print("-" * 50)

    for T in T_list:
        Nc = m2Nc(T, args.me)
        Nv = m2Nv(T, args.mh)
        Ec, Ev = args.Eg, 0.0
        
        # Brent法のブラケット探索
        low, high = -5.0, args.Eg + 5.0
        while deltaQ(low, Ec, Ev, args.ED, args.EA, args.ND, args.NA, T, Nc, Nv) * \
              deltaQ(high, Ec, Ev, args.ED, args.EA, args.ND, args.NA, T, Nc, Nv) > 0:
            low -= 1.0
            high += 1.0
            
        Ef_sol = optimize.brentq(deltaQ, low, high, args=(Ec, Ev, args.ED, args.EA, args.ND, args.NA, T, Nc, Nv))
        
        n_val = Ne(Ef_sol, Ec, T, Nc)
        p_val = Nh(Ef_sol, Ev, T, Nv)
        ndp_val = NDp(Ef_sol, Ec, args.ED, args.ND, T)
        nam_val = NAm(Ef_sol, Ev, args.EA, args.NA, T)

        results["T"].append(T)
        results["Ef"].append(Ef_sol)
        results["n"].append(n_val)
        results["p"].append(p_val)
        results["NDp"].append(ndp_val)
        results["NAm"].append(nam_val)

        if int(T) % 100 == 0 or T == T_list[0]:
            print(f"{T:8.1f} | {Ef_sol:8.4f} | {n_val:10.2e} | {p_val:10.2e}")

    # プロット
    fig, ax1 = plt.subplots(figsize=(6, 6))

    ax1.set_xlabel('Temperature (K)')
    ax1.set_ylabel('Density (cm$^{-3}$)', color='black')
    ax1.semilogy(results["T"], results["n"], 'r-', label='n (electrons)', linewidth=2)
    ax1.semilogy(results["T"], results["p"], 'b-', label='p (holes)', linewidth=2)
    ax1.semilogy(results["T"], results["NDp"], 'r--', label='$N_D^+$ (ionized donors)', alpha=0.6)
    ax1.semilogy(results["T"], results["NAm"], 'b--', label='$N_A^-$ (ionized acceptors)', alpha=0.6)
    ax1.set_ylim(args.Nmin, args.Nmax)
    ax1.grid(True, which="both", ls="-", alpha=0.2)
    ax1.legend(loc='upper left')

    ax2 = ax1.twinx()
    ax2.set_ylabel('Fermi Level $E_f$ (eV)', color='darkgreen')
    ax2.plot(results["T"], results["Ef"], 'g-', label='$E_f$', linewidth=2.5)
    ax2.axhline(args.Eg, color='k', linestyle=':', label='$E_c$')
    ax2.axhline(0, color='k', linestyle=':', label='$E_v$')
    ax2.set_ylim(-0.2, args.Eg + 0.2)
    ax2.legend(loc='upper right')

    plt.title(f'Carrier Statistics (Eg={args.Eg}eV, ND={args.ND:.1e}, NA={args.NA:.1e})')
    fig.tight_layout()
    plt.show()

if __name__ == "__main__":
    main()