"""
Prompt
以下の修正をしてください
# 温度を変えて、Ne,Nh,NDp,NAm,EFをプロットしてください。これもコンソールに出力する
# argparseを使って起動時引数でパラメータを変えられるように。パラメータの初期化にはinitialize()関数を作ってください
# __name__ギミックを使い、main()を呼び出してください
"""


import numpy as np
import matplotlib.pyplot as plt
from scipy.integrate import quad
from scipy.special import expit
from scipy.optimize import brentq
import argparse

# --- 物理定数 ---
kB = 8.61733e-5  # [eV/K]

# --- 物理関数群 ---

def m2Nc(meff, T): return 2.50945e19 * (meff * (T / 300))**(1.5)
def m2Nv(mheff, T): return 2.50945e19 * (mheff * (T / 300))**(1.5)

def Fj(j, eta):
    """j次のフェルミ積分。expitを使用してオーバーフローを回避。"""
    integrand = lambda x: x**j * expit(eta - x)
    res, _ = quad(integrand, 0, max(eta + 50, 100))
    return res

def Ne(Ef, Ec, me, T):
    return m2Nc(me, T) * (2.0 / np.sqrt(np.pi)) * Fj(0.5, (Ef - Ec) / (kB * T))

def Nh(Ef, Ev, mh, T):
    return m2Nv(mh, T) * (2.0 / np.sqrt(np.pi)) * Fj(0.5, (Ev - Ef) / (kB * T))

def NDp(Ef, Ed, ND, T, gD=2):
    return ND * expit(-((Ef - Ed) / (kB * T) + np.log(gD)))

def NAm(Ef, Ea, NA, T, gA=4):
    return NA * expit(-((Ea - Ef) / (kB * T) + np.log(gA)))

def deltaQ(Ef, Ec, Ev, me, mh, ND, Ed, NA, Ea, T):
    return Nh(Ef, Ev, mh, T) - Ne(Ef, Ec, me, T) + NDp(Ef, Ed, ND, T) - NAm(Ef, Ea, NA, T)

def find_Ef(Ec, Ev, me, mh, ND, Ed, NA, Ea, T):
    """電荷中性条件からEfを決定"""
    args = (Ec, Ev, me, mh, ND, Ed, NA, Ea, T)
    low, high = Ev - 0.5, Ec + 0.5
    # 符号反転チェックと範囲拡張
    while deltaQ(low, *args) * deltaQ(high, *args) > 0:
        low -= 0.5
        high += 0.5
    return brentq(deltaQ, low, high, args=args)

# --- 構造化用関数 ---

def initialize():
    """argparseによる引数解析と初期化"""
    parser = argparse.ArgumentParser(description='Semiconductor Temperature Sweep Simulator')
    parser.add_argument('--Eg', type=float, default=1.12, help='Bandgap [eV]')
    parser.add_argument('--me', type=float, default=1.08, help='Electron effective mass ratio')
    parser.add_argument('--mh', type=float, default=0.81, help='Hole effective mass ratio')
    parser.add_argument('--ND', type=float, default=1e17, help='Donor density [cm^-3]')
    parser.add_argument('--NA', type=float, default=1e15, help='Acceptor density [cm^-3]')
    parser.add_argument('--Ed_off', type=float, default=0.045, help='Ec - Ed [eV]')
    parser.add_argument('--Ea_off', type=float, default=0.045, help='Ea - Ev [eV]')
    parser.add_argument('--Tmin', type=float, default=50, help='Min temperature [K]')
    parser.add_argument('--Tmax', type=float, default=600, help='Max temperature [K]')
    return parser.parse_args()

def main():
    args = initialize()
    
    # パラメータ展開
    Ec, Ev = args.Eg, 0.0
    Ed, Ea = Ec - args.Ed_off, Ev + args.Ea_off
    t_list = np.linspace(args.Tmin, args.Tmax, 50)
    
    results = []
    
    print(f"{'T [K]':>6} | {'Ef [eV]':>8} | {'Ne [cm-3]':>9} | {'Nh [cm-3]':>9} | {'ND+ [cm-3]':>9}")
    print("-" * 55)

    for T in t_list:
        ef = find_Ef(Ec, Ev, args.me, args.mh, args.ND, Ed, args.NA, Ea, T)
        ne = Ne(ef, Ec, args.me, T)
        nh = Nh(ef, Ev, args.mh, T)
        ndp = NDp(ef, Ed, args.ND, T)
        nam = NAm(ef, Ea, args.NA, T)
        
        results.append([T, ef, ne, nh, ndp, nam])
        if int(T) % 50 == 0 or T == t_list[0] or T == t_list[-1]:
            print(f"{T:6.1f} | {ef:8.4f} | {ne:9.2e} | {nh:9.2e} | {ndp:9.2e}")

    res = np.array(results).T # 行列転置
    T_arr, Ef_arr, Ne_arr, Nh_arr, NDp_arr, NAm_arr = res

    # --- プロット ---
    fig, ax1 = plt.subplots(figsize=(10, 7))
    
    # キャリア密度プロット (左軸)
    ax1.plot(T_arr, Ne_arr, 'b-', lw=2, label='$n_e$')
    ax1.plot(T_arr, Nh_arr, 'r-', lw=2, label='$n_h$')
    ax1.plot(T_arr, NDp_arr, 'g--', label='$N_D^+$')
    ax1.plot(T_arr, NAm_arr, 'm--', label='$N_A^-$')
    ax1.set_yscale('log')
    ax1.set_xlabel('Temperature [K]')
    ax1.set_ylabel('Density [cm$^{-3}$]')
    ax1.grid(True, which='both', alpha=0.3)
    ax1.legend(loc='upper left')

    # フェルミ準位プロット (右軸)
    ax2 = ax1.twinx()
    ax2.plot(T_arr, Ef_arr, 'k-', lw=3, alpha=0.6, label='$E_F$')
    ax2.axhline(Ec, color='blue', ls=':', alpha=0.4, label='$E_c$')
    ax2.axhline(Ev, color='red', ls=':', alpha=0.4, label='$E_v$')
    ax2.set_ylabel('Energy / Fermi Level [eV]')
    ax2.legend(loc='upper right')

    

    plt.title(f'Temperature dependence ($N_D$={args.ND:.1e}, $N_A$={args.NA:.1e} cm$^{{-3}}$)')
    plt.tight_layout()
    plt.show()

if __name__ == "__main__":
    main()