#Prompt #6
"""
braketが与えられた場合も、念のため探索をしてください。
モデルパラメータ、初期範囲もコンソールに出力して、計算条件を確認できるようにしてください。
deltaQは、log(|deltaQ|)でプロットし、Neなどとは異なるY軸でプロットするように、プログラムを修正してください
"""

import numpy as np
import matplotlib.pyplot as plt
from scipy.integrate import quad
from scipy.special import expit
from scipy.optimize import brentq

# --- 物理定数・基本関数 ---
kB = 8.6173e-5
T_global = 300

def m2Nc(meff, T): return 2.50945e19 * (meff * (T / 300))**(1.5)
def m2Nv(mheff, T): return 2.50945e19 * (mheff * (T / 300))**(1.5)

def Fj(j, eta):
    integrand = lambda x: x**j * expit(eta - x)
    res, _ = quad(integrand, 0, max(eta + 50, 100))
    return res

def Ne(Ef, Ec, meff, T):
    return m2Nc(meff, T) * (2.0 / np.sqrt(np.pi)) * Fj(0.5, (Ef - Ec) / (kB * T))

def Nh(Ef, Ev, mheff, T):
    return m2Nv(mheff, T) * (2.0 / np.sqrt(np.pi)) * Fj(0.5, (Ev - Ef) / (kB * T))

def NDp(Ef, Ed, ND, gD=2, T=300):
    return ND * expit(-((Ef - Ed) / (kB * T) + np.log(gD)))

def NAm(Ef, Ea, NA, gA=4, T=300):
    return NA * expit(-((Ea - Ef) / (kB * T) + np.log(gA)))

def deltaQ(Ef, Ec, Ev, me, mh, ND, Ed, NA, Ea, T):
    return Nh(Ef, Ev, mh, T) - Ne(Ef, Ec, me, T) + NDp(Ef, Ed, ND, T=T) - NAm(Ef, Ea, NA, T=T)

# --- EF決定と範囲探索 ---
def find_Ef_with_check(Ec, Ev, me, mh, ND, Ed, NA, Ea, T, bracket=None):
    args = (Ec, Ev, me, mh, ND, Ed, NA, Ea, T)
    
    # bracketの初期設定
    if bracket is None:
        low, high = Ev, Ec
    else:
        low, high = bracket
    
    # 符号反転が確認できるまで範囲を拡張 (Robustness check)
    step = 0.2
    max_iter = 50
    i = 0
    while deltaQ(low, *args) * deltaQ(high, *args) > 0:
        low -= step
        high += step
        i += 1
        if i > max_iter:
            raise ValueError("解の範囲を特定できませんでした。パラメータを確認してください。")
    
    final_bracket = [low, high]
    ef_sol = brentq(deltaQ, low, high, args=args)
    return ef_sol, final_bracket

# --- メイン処理 ---
# モデルパラメータ
params = {
    'Eg': 1.12, 'me': 1.08, 'mh': 0.81, 'T': 300,
    'ND': 1e17, 'Ed_offset': 0.045, # Ec - Ed
    'NA': 1e15, 'Ea_offset': 0.045  # Ea - Ev
}
Ec, Ev = params['Eg'], 0.0
Ed, Ea = Ec - params['Ed_offset'], Ev + params['Ea_offset']

# EFの計算
ef_eq, search_range = find_Ef_with_check(
    Ec, Ev, params['me'], params['mh'], params['ND'], Ed, params['NA'], Ea, params['T']
)

# 1. コンソールへの条件出力
print("="*40)
print("  SEMICONDUCTOR MODEL PARAMETERS")
print("="*40)
for k, v in params.items(): print(f"{k:<10}: {v}")
print(f"{'Ec':<10}: {Ec} eV / {'Ev':<10}: {Ev} eV")
print("-" * 40)
print(f"Search Bracket : {search_range[0]:.3f} to {search_range[1]:.3f} eV")
print(f"RESULT Ef      : {ef_eq:.6f} eV")
print(f"Final deltaQ   : {deltaQ(ef_eq, Ec, Ev, params['me'], params['mh'], params['ND'], Ed, params['NA'], Ea, params['T']):.2e}")
print("="*40)

# 2. グラフ用データの生成
ef_axis = np.linspace(Ev - 0.2, Ec + 0.2, 500)
ne_plot = [Ne(e, Ec, params['me'], params['T']) for e in ef_axis]
nh_plot = [Nh(e, Ev, params['mh'], params['T']) for e in ef_axis]
dq_plot = [np.log10(np.abs(deltaQ(e, Ec, Ev, params['me'], params['mh'], params['ND'], Ed, params['NA'], Ea, params['T'])) + 1e-20) for e in ef_axis]

# 3. プロット (2軸)
fig, ax1 = plt.subplots(figsize=(10, 6))

# 左軸: キャリア密度
ax1.plot(ef_axis, ne_plot, label='$N_e$', color='blue', lw=2)
ax1.plot(ef_axis, nh_plot, label='$N_h$', color='red', lw=2)
ax1.set_yscale('log')
ax1.set_xlabel('Fermi Level $E_F$ [eV]')
ax1.set_ylabel('Carrier Density [cm$^{-3}$]', color='black')
ax1.grid(True, which='both', alpha=0.3)
ax1.set_ylim(1e10, 1e21)

# 右軸: log|deltaQ|
ax2 = ax1.twinx()
ax2.plot(ef_axis, dq_plot, label=r'$\log_{10}|\Delta Q|$', color='green', ls='--')
ax2.set_ylabel(r'$\log_{10}|\Delta Q|$', color='green')
ax2.tick_params(axis='y', labelcolor='green')

# 垂直線 (バンド端と計算されたEf)
ax1.axvline(Ev, color='k', ls=':', alpha=0.5)
ax1.axvline(Ec, color='k', ls=':', alpha=0.5)
ax1.axvline(ef_eq, color='orange', ls='-', lw=2, label=f'Equil. $E_F$={ef_eq:.3f}eV')



# 凡例の統合
lines, labels = ax1.get_legend_handles_labels()
lines2, labels2 = ax2.get_legend_handles_labels()
ax1.legend(lines + lines2, labels + labels2, loc='upper center', bbox_to_anchor=(0.5, -0.15), ncol=4)

plt.title('Carrier Distribution and Charge Neutrality Convergence')
plt.tight_layout()
plt.show()
