"""
Prompt
EFを決めるコードを追加してください。まず、コードは出力せず、どのようなアルゴリズムを使うか、理由とともに教えてください

Prompt
brent法を使ったコードを実装してください。EFの初期範囲ですが、初期値をNoneあるいは値を与え、初期範囲探索を行ってください
"""


import numpy as np
import matplotlib.pyplot as plt
from scipy.integrate import quad
from scipy.special import expit
from scipy.optimize import brentq

# --- 物理定数・基本関数 (前回の定義を継承) ---
kB = 8.6173e-5
T = 300

def m2Nc(meff, T=300): return 2.50945e19 * (meff * (T / 300))**(1.5)
def m2Nv(mheff, T=300): return 2.50945e19 * (mheff * (T / 300))**(1.5)

def Fj(j, eta):
    integrand = lambda x: x**j * expit(eta - x)
    res, _ = quad(integrand, 0, max(eta + 50, 100))
    return res

def Ne(Ef, Ec, meff, T=300):
    return m2Nc(meff, T) * (2.0 / np.sqrt(np.pi)) * Fj(0.5, (Ef - Ec) / (kB * T))

def Nh(Ef, Ev, mheff, T=300):
    return m2Nv(mheff, T) * (2.0 / np.sqrt(np.pi)) * Fj(0.5, (Ev - Ef) / (kB * T))

def NDp(Ef, Ed, ND, gD=2, T=300):
    return ND * expit(-((Ef - Ed) / (kB * T) + np.log(gD)))

def NAm(Ef, Ea, NA, gA=4, T=300):
    return NA * expit(-((Ea - Ef) / (kB * T) + np.log(gA)))

# --- 電荷中性条件の関数 ---
def deltaQ(Ef, Ec, Ev, me, mh, ND, Ed, NA, Ea, T):
    n = Ne(Ef, Ec, me, T)
    p = Nh(Ef, Ev, mh, T)
    ndp = NDp(Ef, Ed, ND, T=T)
    nam = NAm(Ef, Ea, NA, T=T)
    return p - n + ndp - nam

# --- EFを決定する関数 ---
def find_Ef(Ec, Ev, me, mh, ND, Ed, NA, Ea, T=300, bracket=None):
    """Brent法を用いて電荷中性条件を満たすEfを探索する"""
    
    # 共通引数をパッケージ化
    args = (Ec, Ev, me, mh, ND, Ed, NA, Ea, T)
    
    if bracket is None:
        # 初期範囲の自動探索
        # バンドギャップ周辺から外側へ探索範囲を広げる
        low, high = Ev - 0.5, Ec + 0.5
        step = 0.2
        # 符号が異なる範囲が見つかるまでループ
        while deltaQ(low, *args) * deltaQ(high, *args) > 0:
            low -= step
            high += step
            if high > 10: # 無限ループ防止
                raise ValueError("解の範囲を特定できませんでした。")
        bracket = [low, high]
    
    # Brent法による解探索
    ef_sol = brentq(deltaQ, bracket[0], bracket[1], args=args)
    return ef_sol

# --- メイン処理 ---
Eg, me, mh = 1.12, 1.08, 0.81
Ec, Ev = Eg, 0.0
ND, Ed = 1e17, Eg - 0.045
NA, Ea = 1e15, 0.045

# 1. Efの決定
ef_eq = find_Ef(Ec, Ev, me, mh, ND, Ed, NA, Ea)

# 2. 決定したEfでの各密度を計算して表示
n_eq = Ne(ef_eq, Ec, me)
p_eq = Nh(ef_eq, Ev, mh)
print(f"--- Equilibrium State at T={T}K ---")
print(f"Calculated Fermi Level (Ef): {ef_eq:.4f} eV")
print(f"Electron density (n):       {n_eq:.2e} cm^-3")
print(f"Hole density (p):           {p_eq:.2e} cm^-3")
print(f"Charge Balance (p-n+ND+-NA-): {deltaQ(ef_eq, Ec, Ev, me, mh, ND, Ed, NA, Ea, T):.2e}")

# 3. グラフ作成
ef_range = np.linspace(ef_eq - 0.3, ef_eq + 0.3, 200)
dq_range = [deltaQ(e, Ec, Ev, me, mh, ND, Ed, NA, Ea, T) for e in ef_range]

plt.figure(figsize=(8, 5))
plt.plot(ef_range, dq_range, label=r'Net Charge $\Delta Q(E_F)$', color='blue')
plt.axhline(0, color='red', lw=1)
plt.axvline(ef_eq, color='black', ls='--', label=f'Equilibrium $E_F$ = {ef_eq:.3f} eV')
plt.xlabel('Fermi Level $E_F$ [eV]')
plt.ylabel(r'Net Charge $\Delta Q$ [cm$^{-3}$]')
plt.title('Finding Equilibrium Fermi Level (Brent Method)')
plt.grid(alpha=0.3)
plt.legend()
plt.show()
