"""
Prompt
価電子帯／正孔の関数と、ドナー・アクセプター準位・密度、バンドギャップEgを追加し、
EFを変えながらNh,Ne,イオン化ドナー密度NDp、イオン化アクセプター密度NAmを計算し、
過剰キャリアdeltaQを計算したうえで、コンソールに出力するとともにグラフを描くプログラムに修正してください
"""


import numpy as np
import matplotlib.pyplot as plt
from scipy.integrate import quad
from scipy.special import expit

# --- 物理定数 ---
kB = 8.6173e-5     # [eV/K]
T = 300            # [K]

# --- 基本関数群 ---

def m2Nc(meff, T=300):
    """有効質量比から有効状態密度 Nc [cm^-3] を計算"""
    return 2.50945e19 * (meff * (T / 300))**(1.5)

def m2Nv(mheff, T=300):
    """有効質量比から有効状態密度 Nv [cm^-3] を計算"""
    return 2.50945e19 * (mheff * (T / 300))**(1.5)

def m2Dc0(meff):
    """状態密度関数の前置因子 Dc0 [eV^-1 cm^-3]"""
    return 6.812e21 * (meff)**(1.5)

def fe(E, Ef, T=300):
    """電子のフェルミ・ディラック分布"""
    return expit((Ef - E) / (kB * T))

def fh(E, Ef, T=300):
    """正孔のフェルミ・ディラック分布 (1 - fe)"""
    return expit((E - Ef) / (kB * T))

def Fj(j, eta):
    """j次のフェルミ積分関数"""
    integrand = lambda x: x**j * expit(eta - x)
    upper = max(eta + 50, 100)
    res, _ = quad(integrand, 0, upper)
    return res

def Ne(Ef, Ec, meff, T=300):
    """電子密度 Ne [cm^-3]"""
    eta = (Ef - Ec) / (kB * T)
    return m2Nc(meff, T) * (2.0 / np.sqrt(np.pi)) * Fj(0.5, eta)

def Nh(Ef, Ev, mheff, T=300):
    """正孔密度 Nh [cm^-3]"""
    eta = (Ev - Ef) / (kB * T)
    return m2Nv(mheff, T) * (2.0 / np.sqrt(np.pi)) * Fj(0.5, eta)

def NDp(Ef, Ed, ND, gD=2, T=300):
    """イオン化ドナー密度 N_D+ = ND * [1 / (1 + gD * exp((Ef-Ed)/kT))]"""
    # expitの形に変形: 1 / (1 + exp(x + ln(gD)))
    arg = (Ef - Ed) / (kB * T) + np.log(gD)
    return ND * expit(-arg)

def NAm(Ef, Ea, NA, gA=4, T=300):
    """イオン化アクセプター密度 N_A- = NA * [1 / (1 + gA * exp((Ea-Ef)/kT))]"""
    arg = (Ea - Ef) / (kB * T) + np.log(gA)
    return NA * expit(-arg)

# --- パラメータ設定 (例: Si) ---
Eg = 1.12          # バンドギャップ [eV]
Ec = Eg            # 伝導帯下端
Ev = 0.0           # 価電子帯上端
me = 1.08          # 電子有効質量比
mh = 0.81          # 正孔有効質量比

ND_val = 1e17      # ドナー密度 [cm^-3]
Ed = Ec - 0.045    # ドナー準位 (P in Si)
NA_val = 1e16      # アクセプター密度 [cm^-3]
Ea = Ev + 0.045    # アクセプター準位 (B in Si)

# --- 計算実行 ---
ef_range = np.linspace(-0.6, Ec + 1.0, 400)

ne_list, nh_list, ndp_list, nam_list = [], [], [], []

for ef in ef_range:
    ne_list.append(Ne(ef, Ec, me, T))
    nh_list.append(Nh(ef, Ev, mh, T))
    ndp_list.append(NDp(ef, Ed, ND_val, T=T))
    nam_list.append(NAm(ef, Ea, NA_val, T=T))

ne = np.array(ne_list)
nh = np.array(nh_list)
ndp = np.array(ndp_list)
nam = np.array(nam_list)

# 過剰電荷 deltaQ = p - n + ND+ - NA-
deltaQ = nh - ne + ndp - nam

# --- コンソール出力 (一部) ---
print(f"{'Ef [eV]':>10} | {'Ne [cm^-3]':>10} | {'Nh [cm^-3]':>10} | {'deltaQ':>10}")
print("-" * 50)
for i in range(0, len(ef_range), 40):
    print(f"{ef_range[i]:10.3f} | {ne[i]:10.2e} | {nh[i]:10.2e} | {deltaQ[i]:10.2e}")

# --- プロット ---
plt.figure(figsize=(10, 8))

# 各キャリア・イオン密度のプロット
plt.plot(ef_range, ne, label='$N_e$', lw=2)
plt.plot(ef_range, nh, label='$N_h$', lw=2)
plt.plot(ef_range, ndp, label='$N_D^+$ (Ionized Donor)', ls='--')
plt.plot(ef_range, nam, label='$N_A^-$ (Ionized Acceptor)', ls='--')

# 電荷中性条件の確認用 (絶対値でプロット)
plt.plot(ef_range, np.abs(deltaQ), label=r'$|\Delta Q|$ (Net Charge)', color='black', lw=1, alpha=0.5)

# バンド端の表示
plt.axvline(x=Ev, color='gray', linestyle=':', label='$E_v$')
plt.axvline(x=Ec, color='gray', linestyle=':', label='$E_c$')



plt.yscale('log')
plt.ylim(1e10, 1e22)
plt.xlabel('Fermi Level $E_F$ [eV] (Reference: $E_v=0$)')
plt.ylabel('Density [cm$^{-3}$]')
plt.title('Carrier Densities and Charge Neutrality')
plt.grid(True, which="both", ls="-", alpha=0.2)
plt.legend(bbox_to_anchor=(1.05, 1), loc='upper left')
plt.tight_layout()
plt.show()