"""
Prompt #6
以下の関数を作って書き直してください。関数名は短くしたいので、以下の仕様に従ってください
# 有効質量meff (自由電子質量に対する比)から有効電子密度Ncを計算するm2Nc()
# 有効質量meff (自由電子質量に対する比)から状態密度関数の全治因子Dc0を計算m2Dc0()
#電子のFermi-Dirac関数fe()
#べき乗jのFermi積分関数Fj()
#状態密度関数Dc()
#電子密度を計算する関数 Ne()
"""


import numpy as np
import matplotlib.pyplot as plt
from scipy.integrate import quad
from scipy.special import expit

# --- 物理定数 (単位: eV, cm, K, s, kg) ---
kB = 8.6173e-5     # ボルツマン定数 [eV/K]
m0 = 9.109e-31     # 自由電子質量 [kg]
h = 6.626e-34      # プランク定数 [J·s]
q = 1.602e-19      # 電荷素量 [C]
T = 300            # 温度 [K]

def m2Nc(meff, T=300):
    """有効質量比から有効状態密度 Nc [cm^-3] を計算"""
    # Nc = 2 * (2 * pi * m* * kT / h^2)^(3/2)
    # 単位換算を含めた定数: 2.509e19 * (meff * T / 300)^(3/2)
    return 2.50945e19 * (meff * (T / 300))**(1.5)

def m2Dc0(meff):
    """有効質量比から状態密度関数の前置因子 Dc0 [eV^-1 cm^-3] を計算"""
    # Dc(E) = Dc0 * sqrt(E - Ec)
    # Dc0 = (1/2*pi^2) * (2*m*/hbar^2)^(3/2)
    return 6.812e21 * (meff)**(1.5)

def fe(E, Ef, T=300):
    """電子のフェルミ・ディラック分布関数"""
    eta_x = (Ef - E) / (kB * T)
    return expit(eta_x)

def Fj(j, eta):
    """j次のフェルミ積分関数 Fj(eta)"""

    integrand = lambda x: x**j * expit(eta - x)
    # expit(eta - x) が 1e-18 以下になる点を上限にする
    upper = max(eta + 50, 100)
    res, _ = quad(integrand, 0, upper)
    return res

def Dc(E, Ec, meff):
    """状態密度関数 Dc(E)"""

    dc0 = m2Dc0(meff)
    return dc0 * np.sqrt(np.maximum(0, E - Ec))

def Ne(Ef, Ec, meff, T=300):
    """電子密度 Ne [cm^-3] を計算"""

    nc = m2Nc(meff, T)
    eta = (Ef - Ec) / (kB * T)
    # Ne = Nc * (2/sqrt(pi)) * F_{1/2}(eta)
    return nc * (2.0 / np.sqrt(np.pi)) * Fj(0.5, eta)

# --- メイン処理 ---

meff_si = 1.08  # シリコンの有効質量比 (例)
Ec = 0.0        # 伝導帯下端を基準
ef_ec_range = np.linspace(-0.6, 1.0, 300)

# 計算実行
ne_values = [Ne(diff, Ec, meff_si, T) for diff in ef_ec_range]

# --- プロット ---
plt.figure(figsize=(8, 6))
plt.plot(ef_ec_range, ne_values, label=f'$m_{{eff}}={meff_si}m_0$', color='blue', lw=2)

plt.yscale('log')
plt.title('Electron Density $N_e$ calculation', fontsize=14)
plt.xlabel('$E_F - E_C$ [eV]')
plt.ylabel('$N_e$ [cm$^{-3}$]')
plt.grid(True, which="both", ls="-", alpha=0.2)
plt.axvline(x=0, color='red', linestyle='--')
plt.legend()
plt.tight_layout()
plt.show()