"""
Prompt #3
EF>0でNeが1e20程度で飽和しています。計算の近似が悪いのでは
"""

import numpy as np
import matplotlib.pyplot as plt
from scipy.integrate import quad

# 定数
k_B = 8.6173e-5  # ボルツマン定数 [eV/K]
T = 300           # 温度 [K]
Nc = 2.8e19       # 有効状態密度 [cm^-3] (Siの300Kでの目安)

def fermi_dirac_integral_half(eta):
    """
    フェルミ・ディラック積分 F_{1/2}(eta) の数値計算
    """
    integrand = lambda x: np.sqrt(x) / (1 + np.exp(x - eta))
    # 数値積分を実行
    res, _ = quad(integrand, 0, np.inf)
    return res

def calculate_ne_accurate(ef_ec_range, T, Nc):
    kt = k_B * T
    ne_list = []
    for diff in ef_ec_range:
        eta = diff / kt
        # キャリア密度の基本式: Ne = Nc * (2/√π) * F_{1/2}(eta)
        f_half = fermi_dirac_integral_half(eta)
        ne = Nc * (2.0 / np.sqrt(np.pi)) * f_half
        ne_list.append(ne)
    return np.array(ne_list)

# 1. データの準備 (-0.6 eV から 1.0 eV)
ef_ec_values = np.linspace(-0.6, 1.0, 200)

# 2. 正確な Ne の計算
ne_values = calculate_ne_accurate(ef_ec_values, T, Nc)

# 3. プロット
plt.figure(figsize=(9, 6))
plt.plot(ef_ec_values, ne_values, label='Fermi-Dirac Statistics (Integrated)', color='blue', lw=2)

# 比較用：ボルツマン近似 (Ne = Nc * exp(eta))
ne_boltzmann = Nc * np.exp(ef_ec_values / (k_B * T))
plt.plot(ef_ec_values, ne_boltzmann, '--', label='Boltzmann Approximation', color='gray', alpha=0.6)

plt.yscale('log')
plt.ylim(1e10, 1e23)  # 広い範囲を表示
plt.title('Electron Density $N_e$ vs ($E_F - E_C$)', fontsize=14)
plt.xlabel('$E_F - E_C$ [eV]', fontsize=12)
plt.ylabel('$N_e$ [cm$^{-3}$]', fontsize=12)
plt.grid(True, which="both", ls="-", alpha=0.3)
plt.axvline(x=0, color='red', linestyle=':', label='$E_F = E_C$')
plt.legend()

plt.tight_layout()
plt.show()