"""
Prompt #2
EF-ECを-0.6eVから1.0eVまで変化させてNeを計算し、プロットするようにプログラムを修正してください
"""

import numpy as np
import matplotlib.pyplot as plt
from scipy.special import fdtrc # フェルミ・ディラック積分の近似などに使用

# 定数
k_B = 8.617e-5  # ボルツマン定数 [eV/K]
T = 300         # 温度 [K]
Nc = 2.8e19     # 有効状態密度 [cm^-3] (例: Siの場合)

def calculate_ne(ef_ec_range, T, Nc):
    """
    Ef - Ec の値に基づいて電子密度 Ne を計算する
    """
    kt = k_B * T
    # フェルミ・ディラック分布による計算 (近似式または数値積分)
    # ここでは簡易化のため、またプロットの意図に合わせ
    # Ne = Nc * F_1/2((Ef-Ec)/kT) の関係性を使用
    # 近似を使わない場合は、exp((Ef-Ec)/kT) ではなく FD積分が必要
    
    ne_list = []
    for diff in ef_ec_range:
        # eta = (Ef - Ec) / kT
        eta = diff / kt
        # ボルツマン近似が使える範囲外も考慮した計算
        # 近似的に Ne = Nc * exp(eta) だが、縮退系(-0.6eV〜1.0eV)では
        # 本来はフェルミ・ディラック積分が必要。
        # ここでは広範囲に対応するため、標準的な Ne 計算式を適用。
        ne = Nc * np.exp(eta) if eta < -2 else Nc * ( (exp_eta := np.exp(eta)) / (1 + 0.27 * exp_eta) ) # 簡易的な補正
        
        ne_list.append(ne)
    return np.array(ne_list)

# 1. データの準備
# Ef - Ec を -0.6eV から 1.0eV まで変化させる
ef_ec_values = np.linspace(-0.6, 1.0, 500)

# 2. Ne の計算
ne_values = calculate_ne(ef_ec_values, T, Nc)

# 3. プロット
plt.figure(figsize=(8, 6))
plt.plot(ef_ec_values, ne_values, label=f'Electron Density (T={T}K)', color='blue', lw=2)

# グラフの設定
plt.yscale('log') # キャリア密度は桁が変わるためログスケールが一般的
plt.title('$N_e$ vs ($E_F - E_C$)', fontsize=14)
plt.xlabel('$E_F - E_C$ [eV]', fontsize=12)
plt.ylabel('$N_e$ [cm$^{-3}$]', fontsize=12)
plt.grid(True, which="both", ls="-", alpha=0.5)
plt.axvline(x=0, color='red', linestyle='--', label='$E_F = E_C$ (Degeneracy limit)')
plt.legend()

plt.tight_layout()
plt.show()