"""
Prompt
自由電子モデルで、温度T,Fermi準位EF、有効質量meffを与えてキャリア密度を計算するpythonプログラムをつくってください
"""

import numpy as np
from scipy.integrate import quad
from scipy.constants import k, hbar, eV, m_e

def calculate_carrier_density(T, Ef_ev, meff_ratio, Ec_ev=0):
    """
    キャリア密度を計算する
    :param T: 温度 [K]
    :param Ef_ev: フェルミ準位 [eV]
    :param meff_ratio: 有効質量比 (meff / m_e)
    :param Ec_ev: 伝導帯下端のエネルギー [eV] (デフォルト 0)
    :return: キャリア密度 [cm^-3]
    """
    # 物理定数の準備
    m_eff = meff_ratio * m_e
    Ef = Ef_ev * eV
    Ec = Ec_ev * eV

    # 状態密度 D(E) = (1/2*pi^2) * (2*m_eff/hbar^2)^(3/2) * sqrt(E - Ec)
    def dos(E):
        if E < Ec:
            return 0
        return (1 / (2 * np.pi**2)) * (2 * m_eff / hbar**2)**1.5 * np.sqrt(E - Ec)

    # フェルミ・ディラック分布 f(E)
    def fermi_dist(E):
        if T == 0:
            return 1.0 if E <= Ef else 0.0
        
        # 指数関数のオーバーフロー対策
        res = (E - Ef) / (k * T)
        if res > 40: return 0.0
        if res < -40: return 1.0
        return 1 / (np.exp(res) + 1)

    # 被積分関数 D(E) * f(E)
    def integrand(E):
        return dos(E) * fermi_dist(E)

    # 数値積分 (Ec から Ef + 20kbT 程度まで積分すれば十分)
    limit_upper = Ef + 20 * k * T if T > 0 else Ef
    density_m3, error = quad(integrand, Ec, max(limit_upper, Ec + 1*eV))

    # m^-3 から cm^-3 に変換
    density_cm3 = density_m3 * 1e-6
    return density_cm3

# --- 設定値 ---
T_room = 300            # 温度 [K]
Ef_input = 2.0          # フェルミ準位 [eV] (Ec=0 とした時の値)
meff_input = 1.0        # 有効質量比 (電子質量に対する比)

n = calculate_carrier_density(T_room, Ef_input, meff_input)

print(f"Temperature: {T_room} K")
print(f"Fermi Level: {Ef_input} eV")
print(f"Effective Mass: {meff_input} m_e")
print(f"Carrier Density: {n:.4e} cm^-3")