import sys
import random
import numpy as np
from scipy.optimize import minimize
import matplotlib.pyplot as plt
from mpl_toolkits.mplot3d import Axes3D


method = "ga"

# 変数の範囲
bounds = [(-10, 10), (-10, 10)]
#初期値 (simplex)
initial_guess = [5, 5]

# 母集団の個体数
nparents = 50

nmaxiter = 100


nargs = len(sys.argv)
if nargs > 1: method = sys.argv[1]
if nargs > 2: nparents = int(sys.argv[2])
if nargs > 3: nmaxiter = int(sys.argv[3])


def objective_function(x):
    X, Y = x[0], x[1]
#    return (x[0] - 1.0)**2 + (x[1] + 2)**2

# Ackley function
    t1 = 20
    t2 = -20 * np.exp(-0.2 * np.sqrt(1.0 / 2 * (X**2 + Y**2 )))
    t3 = np.e
    t4 = -np.exp(1.0 / 2 * (np.cos(2 * np.pi * X)+np.cos(2 * np.pi * Y)))
    return t1 + t2 + t3 + t4

def usage():
    print(f"\nUsage: python {sys.argv[0]} method=[ga|sworm|remc|simplex] nparents nmaxiter\n")
def create_individual(bounds):
    return np.array([random.uniform(low, high) for low, high in bounds])


#######################
# GA
#######################
def mutate(individual, bounds, mutation_rate = 0.01):
    for i in range(len(individual)):
        if random.random() < mutation_rate:
            individual[i] = random.uniform(bounds[i][0], bounds[i][1])

def crossover(parent1, parent2):
    crossover_point = random.randint(1, len(parent1) - 1)
    child1 = np.concatenate((parent1[:crossover_point], parent2[crossover_point:]))
    child2 = np.concatenate((parent2[:crossover_point], parent1[crossover_point:]))
    return child1, child2

def select_best_by_random_group(population, scores, k = 3):
    selected = random.choices(population, k = k)

# selectedから、scoresに対応するindexを検索してscoresに入れる
#    selected_scores = [scores[i] for i, ind in enumerate(population) if any((ind == s).all() for s in selected)]
    selected_scores = []
    selected_params = []
    for i, ind in enumerate(population):
        for s in selected:
            if (ind == s).all():
                selected_scores.append(scores[i])
                selected_params.append(s)
                break
#    if not selected_scores:  # selected_scoresが空の場合の対処
#        return random.choice(population)

    idx = np.argmin(selected_scores)
    return selected_params[idx]

def genetic_algorithm(objective_function, bounds, population_size, generations, mutation_rate = 0.01):
    population = [create_individual(bounds) for _ in range(population_size)]
    best_individual = population[0]
    best_score = objective_function(best_individual)

    for generation in range(generations):
        scores = [objective_function(ind) for ind in population]
        new_population = []
        for i in range(population_size // 2):
            parent1 = select_best_by_random_group(population, scores)
            parent2 = select_best_by_random_group(population, scores)
            child1, child2 = crossover(parent1, parent2)
            mutate(child1, bounds, mutation_rate)
            mutate(child2, bounds, mutation_rate)
            new_population.extend([child1, child2])

        population = new_population
        current_best = min(population, key=objective_function)
        current_best_score = objective_function(current_best)
        if current_best_score < best_score:
            best_individual, best_score = current_best, current_best_score

    return best_individual, best_score


##########################
# sworm optimization
##########################
class Particle:
    def __init__(self, bounds):
        self.position = np.array([np.random.uniform(low, high) for low, high in bounds])
        self.velocity = np.array([0.0 for _ in bounds])
        self.best_position = self.position.copy()
        self.best_score = float('inf')

    def update_velocity(self, global_best_position, inertia, cognitive, social):
        r1, r2 = np.random.rand(2)
        cognitive_velocity = cognitive * r1 * (self.best_position - self.position)
        social_velocity = social * r2 * (global_best_position - self.position)
        self.velocity = inertia * self.velocity + cognitive_velocity + social_velocity

    def update_position(self, bounds):
        self.position += self.velocity
        for i, (low, high) in enumerate(bounds):
            if self.position[i] < low:
                self.position[i] = low
            elif self.position[i] > high:
                self.position[i] = high

def pso(objective_function, bounds, num_particles, max_iter, inertia=0.5, cognitive=1.5, social=1.5):
    particles = [Particle(bounds) for _ in range(num_particles)]
    global_best_position = particles[0].position.copy()
    global_best_score = float('inf')

    for _ in range(max_iter):
        for particle in particles:
            score = objective_function(particle.position)
            if score < particle.best_score:
                particle.best_score = score
                particle.best_position = particle.position.copy()
            if score < global_best_score:
                global_best_score = score
                global_best_position = particle.position.copy()

        for particle in particles:
            particle.update_velocity(global_best_position, inertia, cognitive, social)
            particle.update_position(bounds)

    return global_best_position, global_best_score

##############################
# Replica Exchange MonteCarlo
##############################
def exchange_acceptance(delta, temp1, temp2):
    return np.exp(delta * (1/temp1 - 1/temp2))

def replica_exchange_monte_carlo(objective_function, bounds, num_replicas, num_iterations, temperatures):
    replicas = [create_individual(bounds) for _ in range(num_replicas)]
    best_individual = replicas[0]
    best_score = objective_function(best_individual)

    for iteration in range(num_iterations):
        for i in range(num_replicas):
            new_individual = create_individual(bounds)
            delta = objective_function(new_individual) - objective_function(replicas[i])
            if delta < 0 or np.random.rand() < np.exp(-delta / temperatures[i]):
                replicas[i] = new_individual

        for i in range(num_replicas - 1):
            delta = objective_function(replicas[i+1]) - objective_function(replicas[i])
            if np.random.rand() < exchange_acceptance(delta, temperatures[i], temperatures[i+1]):
                replicas[i], replicas[i+1] = replicas[i+1], replicas[i]

        current_best = min(replicas, key=objective_function)
        current_best_score = objective_function(current_best)
        if current_best_score < best_score:
            best_individual, best_score = current_best, current_best_score

    return best_individual, best_score


def optimize(method):
    if method == 'ga':
        best_params, best_score = genetic_algorithm(objective_function, bounds, nparents, nparents)
    elif method == 'simplex':
        initial_simplex = np.array([
            initial_guess,
            (bounds[0][0], initial_guess[1]),
            (initial_guess[0], bounds[1][1]),
            ])
        result = minimize(objective_function, initial_guess, method = 'Nelder-Mead', options = {'initial_simplex': initial_simplex})
        best_params = result.x
        best_score = result.fun
    elif method == 'sworm':
        best_params, best_score = pso(objective_function, bounds, nparents, nparents)
    elif method == 'remc':
        temperatures = np.linspace(1, 10, nparents)
        best_params, best_score = replica_exchange_monte_carlo(objective_function, bounds, nparents, nmaxiter, temperatures)
    else:
        print(f"Invalide method [{method}]")
        exit()

    print()
    print(f"method: {method}")
    print(f"Best parameters: {best_params}")
    print(f"Best score: {best_score}")

def plot():
    x = np.linspace(-5, 5, 400)
    y = np.linspace(-5, 5, 400)
    x, y = np.meshgrid(x, y)    

    z = objective_function([x, y])

    fig = plt.figure()
    ax = fig.add_subplot(111, projection='3d')
    ax.plot_surface(x, y, z, cmap='viridis')

    ax.set_xlabel('X')
    ax.set_ylabel('Y')
    ax.set_zlabel('Z')

    plt.pause(0.01)
    input("\nPress ENTER to terminate>>\n")


def main():
    print()
    print(f"method: {method}")
    print(f"bounds:", bounds)
    print(f"initial guess:", initial_guess)
    print(f"nmaxiter: {nmaxiter}")

    if method == 'all':
        for m in ['ga', 'sworm', 'remc', 'simplex']:
            optimize(m)
    else:
       optimize(method)


if __name__ == '__main__':
    plot()

    main()
    
    usage()
    
    