import os
import sys
from pydub import AudioSegment

# ====== 設定 ======
output_dir = "split_fixed"
output_format = "mp3"           # 出力形式（"mp3", "wav"など）
bitrate = "128k"                # 出力ビットレート
max_size_mb = 150               # チャンク最大サイズ（MB）
overlap_sec = 60                # オーバーラップ時間（秒）

# ====== 引数処理 ======
if len(sys.argv) < 2:
    print("使い方: python fixed_split.py <input_audio_file>")
    sys.exit(1)

nargs = len(sys.argv)
if nargs > 2: input_file = sys.argv[1]
if nargs > 3: output_format = sys.argv[2]
if nargs > 4: bitrate = sys.argv[3]
if nargs > 5: max_size_mb = float(sys.argv[4])
if nargs > 6: overlap_sec = float(sys.argv[5])

if not os.path.isfile(input_file):
    print(f"エラー: ファイルが見つかりません: {input_file}")
    sys.exit(1)

# ====== 音声読み込み ======
ext = os.path.splitext(input_file)[1].lower().lstrip(".")
print(f"\n音声ファイルを読み込み中...: {input_file} (.{ext})")
audio = AudioSegment.from_file(input_file, format=ext)
duration_sec = len(audio) // 1000

# ====== 処理情報出力 ======
print(f"出力ディレクトリ: {output_dir}")
print(f"出力フォーマット: {output_format}")
print(f"ビットレート: {bitrate}")
print(f"オーバーラップ長: {overlap_sec}秒")
print(f"最大サイズ: {max_size_mb}MB")
print(f"音声長さ: {duration_sec} 秒")

# ====== 出力ディレクトリ準備 ======
dir = os.path.dirname(input_file)
output_dir = os.path.join(dir, output_dir)
print("output_dir:", output_dir)
if not os.path.exists(output_dir):
    os.makedirs(output_dir, exist_ok=True)

# ====== チャンク長（秒）の見積もり ======
kbps = int(bitrate.replace("k", ""))
bytes_per_sec = kbps * 1000 // 8
max_bytes = max_size_mb * 1024 * 1024
max_chunk_sec = max_bytes // bytes_per_sec
step = max_chunk_sec - overlap_sec  # オーバーラップを考慮した前進幅

print(f"推定チャンク長: 約 {max_chunk_sec} 秒（@{bitrate}）")
print(f"ステップ（次の開始位置）: {step} 秒（オーバーラップ {overlap_sec} 秒）")
print(f"\n分割開始: 約{max_chunk_sec}秒ごとに分割（{overlap_sec}秒の重なりあり）")

# ====== 分割処理 ======
chunk_index = 0
start_sec = 0
filebody = os.path.splitext(os.path.basename(input_file))[0]

while start_sec < duration_sec:
    print(f"> チャンク {chunk_index + 1} 開始: {start_sec}秒") 
    end_sec = start_sec + max_chunk_sec
    end_sec = min(end_sec, duration_sec)  # 終端を超えない

    # 切り出し（ミリ秒）
    start_ms = start_sec * 1000
    end_ms = end_sec * 1000
    chunk = audio[start_ms:end_ms]

    # ファイル保存
    out_path = os.path.join(output_dir, f"{filebody}_{chunk_index:03}.{output_format}")
    chunk.export(out_path, format=output_format, bitrate=bitrate)
    print(f"> 保存: {out_path}（{(end_sec - start_sec)} 秒）")

    # 次の位置へ進む（オーバーラップ考慮）
    start_sec += step
    chunk_index += 1

print("\n✅ サイズベースでオーバーラップ付き分割を完了しました。")
input("\nPress ENTER to terminate>>\n")
