#pip install python-pptx

import os
import sys
try:
    from pptx import Presentation
except:
    print("\npptx2pdf_recursive.py: Import error: pptx")
    input("Install: pip install python-pptx\n")


root_dir = "."  # 現在のディレクトリ
output_path = "extracted_pptx_text.txt"
max_level = -1  # 無制限

# python script.py [root_dir] [output_path] [max_level]
if len(sys.argv) > 1: root_dir = sys.argv[1]
if len(sys.argv) > 2: output_path = sys.argv[2]
if len(sys.argv) > 3: max_level = sys.argv[3]
try:
    max_level = int(max_level)
except ValueError:
    print(f"Error: Invalid max_level '{max_level}'.")
    exit()

def extract_text_from_pptx(pptx_path, output_file, mode="paragraph"):
    """
    指定されたPowerPointファイルからテキストを抽出し、出力ファイルに書き込む
    """
    try:
        prs = Presentation(pptx_path)

        for i, slide in enumerate(prs.slides):
            slide_text = []

            # 1. スライドタイトルを最初に抽出
            slide_title_text = ""
            if slide.shapes.title: # 標準のタイトルプレースホルダーがあるか確認
                slide_title_text = slide.shapes.title.text_frame.text.strip()

            for shape in slide.shapes:
                if hasattr(shape, "text_frame") and shape.text_frame:
                    for paragraph in shape.text_frame.paragraphs:
                        if mode == 'paragraph':
                            slide_text.append(paragraph.text)
                        else:
                            for run in paragraph.runs:
                                if run.text:
                                    slide_text.append(run.text)
            
            if slide_text:
                output_file.write(f"{pptx_path}\n")
                if slide_title_text:
                    output_file.write(f"Slide {i + 1}: {slide_title_text.replace('\n', ' ')}\n")
                else:
                    output_file.write(f"Slide {i + 1}:\n")
                output_file.write("".join(slide_text).strip() + "\n\n")
            else:
                # スライドにテキストがない場合でも、ファイルパスとスライド番号は記録する
                output_file.write(f"{pptx_path}\n")
                output_file.write(f"Slide {i + 1}: (No text found)\n\n")

    except Exception as e:
        print(f"Error processing {pptx_path}: {e}")

def search_pptx_and_extract(root_dir, output_path, max_level=-1):
    """
    指定されたディレクトリからPowerPointファイルを再帰的に検索し、テキストを抽出する
    """
    if not os.path.isdir(root_dir):
        print(f"Error: Root directory '{root_dir}' does not exist.")
        return

    print(f"Searching for .pptx files in '{root_dir}' (max_level: {max_level})...")
    
    with open(output_path, 'w', encoding='utf-8') as outfile:
        for dirpath, dirnames, filenames in os.walk(root_dir):
            # 現在のディレクトリの深さを計算
            current_level = dirpath.count(os.sep) - root_dir.count(os.sep)
            
            if max_level != -1 and current_level >= max_level:
                # 指定された深さ制限を超えた場合、このブランチはスキップ
                del dirnames[:] # これにより、os.walkがこのディレクトリの下に進まないようにする
                continue

            for filename in filenames:
                if filename.endswith(".pptx"):
                    pptx_full_path = os.path.join(dirpath, filename)
                    print(f"  Processing: {pptx_full_path}")
                    extract_text_from_pptx(pptx_full_path, outfile)

if __name__ == "__main__":
    search_pptx_and_extract(root_dir, output_path, max_level)
    print(f"\nText extraction complete. Output saved to '{output_path}'")
    input("\nPress ENTER to terminate>>")
    