import os
import sys
import datetime
try:
    import comtypes.client
except:
    print("\npptx2pdf_recursive.py: Import error: comtypes")
    input("Install: pip install comtypes\n")

root_dir = "."  # 現在のディレクトリ
max_level = -1  # 無制限

# python script.py [root_dir] [max_level]
if len(sys.argv) > 1: root_dir = sys.argv[1]
if len(sys.argv) > 2: max_level = sys.argv[2]
try:
    max_level = int(max_level)
except ValueError:
    print(f"Error: Invalid max_level '{max_level}'.")
    exit()


# --- 修正箇所：pptx_to_pdf 関数内 ---
def pptx_to_pdf(input_path, output_path=None):
    """
    PowerPointファイルをPDFに変換します。
    """
    # PowerPointのアプリケーションを起動
    powerpoint = comtypes.client.CreateObject("PowerPoint.Application")
    powerpoint.Visible = 1 # アプリを表示しない場合は 0。以前の議論を踏まえて0のままにしておきます。

    # 出力先のパスを指定
    if output_path is None:
        output_path = os.path.splitext(input_path)[0] + ".pdf"

    print(f"  Attempting to open PowerPoint file: '{input_path}'") # <-- ここを追加
    
    try:
        # プレゼンテーションを開く
        presentation = powerpoint.Presentations.Open(input_path, WithWindow=False)

        # PDFとして保存（フォーマット番号 32 = PDF）
        presentation.SaveAs(output_path, 32)
        presentation.Close()
        print(f"  Successfully converted to PDF: '{output_path}'")
        return output_path
    except Exception as e:
        print(f"  Error converting '{input_path}' to PDF: {e}")
        return None
    finally:
        # PowerPointアプリケーションを終了
        powerpoint.Quit()

def convert_pptx_files(root_dir, max_level=-1):
    """
    指定されたディレクトリ以下のPowerPointファイルを検索し、必要に応じてPDFに変換します。
    """
    # --- 修正箇所：root_dirを絶対パスに変換 ---
    root_dir = os.path.abspath(root_dir) # <-- ここを追加

    if not os.path.isdir(root_dir):
        print(f"Error: Root directory '{root_dir}' does not exist.")
        return

    print(f"Searching for .pptx files in '{root_dir}' (max_level: {max_level})...")
    
    for dirpath, dirnames, filenames in os.walk(root_dir):
        # 現在のディレクトリの深さを計算
        current_level = dirpath.count(os.sep) - root_dir.count(os.sep)
        
        if max_level != -1 and current_level >= max_level:
            # 指定された深さ制限を超えた場合、このブランチはスキップ
            del dirnames[:] # これにより、os.walkがこのディレクトリの下に進まないようにする
            continue

        for filename in filenames:
            if filename.lower().endswith(".pptx"):
                pptx_full_path = os.path.join(dirpath, filename)
                pdf_full_path = os.path.splitext(pptx_full_path)[0] + ".pdf"

                needs_conversion = False

                if not os.path.exists(pdf_full_path):
                    print(f"'{os.path.basename(pdf_full_path)}' does not exist. Converting...")
                    needs_conversion = True
                else:
                    pptx_timestamp = os.path.getmtime(pptx_full_path)
                    pdf_timestamp = os.path.getmtime(pdf_full_path)

                    if pptx_timestamp > pdf_timestamp:
                        print(f"'{os.path.basename(pdf_full_path)}' is older than '{os.path.basename(pptx_full_path)}'. Converting...")
                        needs_conversion = True
                    else:
                        print(f"'{os.path.basename(pptx_full_path)}' is up to date. Skipping conversion.")

                if needs_conversion:
                    pptx_to_pdf(pptx_full_path, pdf_full_path)
    
    print("\nPDF conversion process finished.")

if __name__ == "__main__":
    convert_pptx_files(root_dir, max_level)
    print("\nProgram execution completed.")
    input("\nPress ENTER to terminate>>\n")
    