#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Pandoc専用：MarkdownのWord/PowerPoint/HTML変換とテンプレート生成。
- AI処理は一切ありません（tkai_lib不要）
- 例:
    # テンプレート生成
    python pandoc_convert.py --make_template docx --template template_textbook.docx
    python pandoc_convert.py --make_template pptx --template template_slide.pptx

    # 変換
    python pandoc_convert.py --convert docx --infile lecture_textbook.md --template template_textbook.docx
    python pandoc_convert.py --convert pptx --infile lecture_slide.md --template template_slide.pptx
    python pandoc_convert.py --convert html --infile lecture_textbook.md --mathml
    python pandoc_convert.py --convert html --infile lecture_textbook.md --css custom_style.css
"""

import os
import sys
import argparse
from pathlib import Path
import shutil
import platform
import subprocess
from pathlib import Path


def find_pandoc() -> str | None:
    """
    pandoc 実行ファイルのパスを探す。
    優先順位:
    1. 環境変数 pandoc_path
    2. 環境変数 tkProg_path に基づく既定配置
       - Windows: {tkProg_path}/tkApp_Win/pandoc/pandoc.exe
       - Linux:   {tkProg_path}/tkApp_Linux/pandoc/pandoc
    3. PATH 環境変数から探す
    見つからなければ None を返す。
    """

    # 1. 環境変数 pandoc_path
    env_pandoc = os.environ.get("pandoc_path")
    if env_pandoc and os.path.isfile(env_pandoc):
        return env_pandoc

    # 2. tkProg_path 内
    tkprog = os.environ.get("tkProg_path")
    if tkprog is None:
        script_dir = Path(sys.argv[0]).resolve().parent
        tkprog = (script_dir / "../../").resolve()
    if tkprog:
        if platform.system() == "Windows":
            candidate = os.path.join(tkprog, "tkApp_Win", "pandoc", "pandoc.exe")
        else:
            candidate = os.path.join(tkprog, "tkApp_Linux", "pandoc", "pandoc")
        if os.path.isfile(candidate):
            return candidate

    # 3. PATH から探す
    exe_name = "pandoc.exe" if platform.system() == "Windows" else "pandoc"
    path = shutil.which(exe_name)
    if path:
        return path

    return "pandoc"

def find_template(template_path: str) -> str | None:
    """
    template_path で与えられたファイル名を以下の順に探す:
    1. 実行ディレクトリ (カレントディレクトリ)
    2. スクリプトのあるディレクトリ (sys.argv[0] 基準)

    見つかればフルパスを返し、見つからなければ None を返す。
    """
    if not template_path:
        return None

    # 1. 実行ディレクトリ（カレント）
    candidate1 = Path.cwd() / template_path
    if candidate1.is_file():
        return str(candidate1.resolve())

    # 2. スクリプトのあるディレクトリ
    script_dir = Path(sys.argv[0]).resolve().parent
    candidate2 = script_dir / template_path
    if candidate2.is_file():
        return str(candidate2.resolve())

    # 見つからなければ None
    return template_path


def parse_args():
    pandoc_path = find_pandoc()
    print(f"✅ pandoc found: {pandoc_path}")

    p = argparse.ArgumentParser(
        description="Pandocユーティリティ（テンプレート生成 & 変換専用）",
        formatter_class=argparse.RawTextHelpFormatter
    )
    p.add_argument('--pandoc_path', default=pandoc_path, help=f'pandoc 実行ファイルのパス (デフォルト {pandoc_path})')
    p.add_argument('--infile', help='変換元Markdown（--convert使用時に指定）')
    p.add_argument('--convert', choices=['docx', 'pptx', 'html'], help='Pandoc変換を実行')
    p.add_argument('--template', default = None, help='変換に使用、または作成するテンプレートファイルのパス')
    p.add_argument('--mathml', action='store_true', help='HTML変換時にMathMLを使って数式を表示')
    p.add_argument('--css', help='HTML変換時のカスタムCSS')
    p.add_argument('--make_template', choices=['docx', 'pptx'], help='デフォルトテンプレートを出力して終了')
    p.add_argument('--no-yaml', action='store_true', help='入力を markdown-yaml_metadata_block として解釈する')
    p.add_argument('--verbose', action='store_true', help='詳細出力')
    p.add_argument('--pause', action='store_true', help='終了前に入力待ち')
    args = p.parse_args()

    if args.template:
        args.template = find_template(args.template)
        print(f"✅ template found: {args.template}")

    return p, args

def run_command(cmd: list) -> bool:
    print(f"⚡️ 実行中: {' '.join(cmd)}")
    try:
        _ = subprocess.run(cmd, check=True, capture_output=True, text=True, encoding='utf-8')
        print("✅ コマンドは正常に完了しました。")
        return True
    except FileNotFoundError:
        print(f"❌ '{cmd[0]}' が見つかりません。Pandocはインストール済みでPATHが通っていますか？", file=sys.stderr)
    except subprocess.CalledProcessError as e:
        print(f"❌ Pandoc実行エラー (終了コード: {e.returncode})", file=sys.stderr)
        print(f"--- stderr ---\n{e.stderr}\n--------------", file=sys.stderr)
    return False

def make_pandoc_template(pandoc_path: str, fmt: str, template_file: str):
    print(f"🎨 {fmt.upper()} 用のPandocテンプレートを作成します...")
    data_file = 'reference.docx' if fmt == 'docx' else 'reference.pptx'
    cmd = [pandoc_path, '-o', template_file, f'--print-default-data-file={data_file}']
    
    if run_command(cmd):
        print(f"📄 テンプレート '{template_file}' を作成しました。編集してスタイルを調整してください。")

def correct_html(outfile):
    with open(outfile, 'r', encoding='utf-8') as f:
        body_content = f.read()

# MathJax対応のHTMLとしてラップ
    html_template = f"""<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <script src="https://polyfill.io/v3/polyfill.min.js?features=es6"></script>
  <script id="MathJax-script" async
    src="https://cdn.jsdelivr.net/npm/mathjax@3/es5/mml-chtml.js"></script>
</head>

<body>
{body_content}
</body>
</html>
"""

    with open(outfile, 'w', encoding='utf-8') as f:
        f.write(html_template)

def convert_with_pandoc(pandoc_path: str, target: str, infile_md: str, template: str = None, css: str = None, no_yaml: bool = False, args = None):
    print(f"🔄 '{infile_md}' → {target.upper()} 変換を開始します...")
    if not Path(infile_md).exists():
        print(f"❌ 入力Markdown '{infile_md}' が存在しません。", file=sys.stderr)
        sys.exit(1)

    outfile = str(Path(infile_md).with_suffix(f'.{target}'))
    # ベース引数
    if no_yaml:
        cmd = [pandoc_path, infile_md, '-o', outfile, '-f', 'markdown-yaml_metadata_block']
    else:
        cmd = [pandoc_path, infile_md, '-o', outfile]

    if args.mathml:
        cmd.append('--mathml')
    if args.verbose:
        cmd.append('--verbose')

    # 出力形式ごとの追加
    if target == 'html':
        if css:
            cmd.extend(['--css', css])
    elif target in ('docx', 'pptx'):
        if not template:
            pass
#            print("❌ --convert docx/pptx では --template が必須です。", file=sys.stderr)
#            sys.exit(1)
        else:
            cmd.extend(['--reference-doc', template])

#        if target == 'pptx':
#            cmd.extend(['-t', 'pptx'])
        cmd.extend(['-t', target])

    ret = run_command(cmd)
    if ret:
        if args.convert == 'html':
            correct_html(outfile)
        
        print(f"📑 変換後のファイル '{outfile}' を作成しました。")

def main():
    parser, args = parse_args()

    # 1) テンプレート生成（最優先）
    if args.make_template:
        if not args.template:
            parser.error("--make_template には --template で出力先ファイル名が必要です。")
        make_pandoc_template(args.pandoc_path, args.make_template, args.template)
        if args.pause:
            input("\nPress ENTER to exit >>\n")
        return

    # 2) 変換
    if args.convert:
        if not args.infile:
            parser.error("--convert を使うときは --infile（Markdown）を指定してください。")
        convert_with_pandoc(
            args.pandoc_path,
            args.convert,
            args.infile,
            template=args.template,
            css=args.css,
            no_yaml=args.no_yaml,
            args = args,
        )
        if args.pause:
            input("\nPress ENTER to exit >>\n")
        return

    # 3) どちらも指定がない場合
    parser.print_help()

if __name__ == "__main__":
    main()
