import os
import sys
import argparse
import glob
import convert2md
import pptx2pdf
import pptx2img
import pptx2md
import docx2img
import pdf2img
import pdf2md
import img2pdf


# 引数の設定とパースを行う関数
def initialize():
    """
    コマンドライン引数を解析し、ArgumentParserオブジェクトを返す
    """
    # ヘルプメッセージ付きのパーサーを作成
    parser = argparse.ArgumentParser(
        description='PowerPointファイルを画像に変換します。',
        formatter_class=argparse.RawTextHelpFormatter
    )

    # 引数を追加
    parser.add_argument(
        '--infile',
        default='41-01-search_MP.pptx',
        help='入力ファイル(pptx/docx/pdf)のパスを指定します。\n'
             '(デフォルト: 41-01-search_MP.pptx)'
    )
    
    parser.add_argument(
        '--output_dir',
        default='images',
        help='画像を保存するディレクトリのパスを指定します。\n'
             '(デフォルト: images)'
    )
    
    parser.add_argument(
        '--output_format',
        default='png',
        choices=['png', 'md', 'pdf'],
        help='出力画像のフォーマットを指定します。\n'
             '現在、png/md のみがサポートされています。\n'
             '(デフォルト: png)'
    )
    
    # 引数を解析して返す
    return parser


def main():
    parser = initialize()
    args = parser.parse_args()

    infile = args.infile
    output_dir = args.output_dir
    output_format = args.output_format

    infile_ext = os.path.splitext(infile)[1].lower()
    print()
    print(f"入力ファイル: {infile} (拡張子: {infile_ext})")
    print(f"出力ディレクトリ: {output_dir}")
    print(f"出力フォーマット: {output_format}")
    print(f"入力ファイルタイプ: {infile_ext}")

    if infile_ext.lower() in ['.jpeg', '.jpg', '.png']:
        if output_format == 'pdf':
            img2pdf.convert_image_to_pdf(infile)
    elif infile_ext == ".pptx":
        if output_format == 'png':
            pptx2img.export_img(infile, output_dir, output_format)
            pptx2img.rename_img(output_dir) # out_dirはoutput_dirに変更
        elif output_format == 'pdf':
            filename_without_ext = os.path.splitext(os.path.basename(infile))[0]
            pdf_path = os.path.join(os.path.dirname(infile), f"{filename_without_ext}.pdf")
            pptx2pdf.pptx_to_pdf(infile, pdf_path)
        elif output_format == 'md':
#            pptx2md.convert(infile, None)
            convert2md.convert(infile, None)
        else:
            print(f"\nError: Invalid output_format [{output_format}] for input type [{infile_ext}]\n")
    elif infile_ext == ".docx":
        if output_format == 'png':
            filename_without_ext = os.path.splitext(os.path.basename(infile))[0]
            pdf_path = os.path.join(os.path.dirname(infile), f"{filename_without_ext}.pdf")
            docx2img.convert_word_to_pdf(infile, pdf_path)
            docx2img.convert_pdf_to_images(pdf_path, output_dir)
        elif output_format == 'pdf':
            filename_without_ext = os.path.splitext(os.path.basename(infile))[0]
            pdf_path = os.path.join(os.path.dirname(infile), f"{filename_without_ext}.pdf")
            docx2img.convert_word_to_pdf(infile, pdf_path)
        elif output_format == 'md':
            convert2md.convert(infile, None)
        else:
            print(f"\nError: Invalid output_format [{output_format}] for input type [{infile_ext}]\n")
    elif infile_ext == ".pdf":
        if output_format == 'png':
            pdf2img.convert_pdf_to_images(infile, output_dir, output_format)
        elif output_format == 'md':
            pdf2md.convert(infile, None)
        elif output_format == 'md':
            convert2md.convert(infile, None)
        else:
            print(f"\nError: Invalid output_format [{output_format}] for input type [{infile_ext}]\n")

    input("\n終了するにはENTERキーを押してください>>\n")


if __name__ == '__main__':
    main()