#pip install PyPDF2 python-docx python-pptx reportlab

#次のpythonプログラムを作ってください
#argparseでwork_dirとnmergeをうけとる。それぞれのデフォルトを., 10にする
#ディレクトリ"merged"がなければつくる
#work_dirから*.pdfを検索、ソートし、nmergeファイルごとに連結したPDFファイルを作って
#mergedに保存する。この際、読み込み、連結に失敗したファイルは、そのままmergedにコピー

import argparse
import glob
import os
import shutil
import fnmatch
import io
import re
from fpdf import FPDF
from PyPDF2 import PdfMerger, PdfReader


try:
    from docx import Document
    from docx.shared import Inches
except ImportError:
    print("警告: 'python-docx' がインストールされていません。Wordファイルの処理はできません。")
    print("インストールするには 'pip install python-docx' を実行してください。")
    Document = None

try:
    from pptx import Presentation
    from pptx.util import Inches as PptxInches # pptxのInchesと混同しないように別名でインポート
    from pptx.enum.shapes import MSO_SHAPE_TYPE
except ImportError:
    print("警告: 'python-pptx' がインストールされていません。PowerPointファイルの処理はできません。")
    print("インストールするには 'pip install python-pptx' を実行してください。")
    Presentation = None


"""
複数のPDFファイルをマージする
・どのPDFファイルがマージされたかは、マージファイルの先頭ページに記入
・マージされたファイルが始まるページには、そのファイル名のページを挿入

起動時引数:
  --mode: "rename"の場合、ソースファイル名の先頭に4桁の連番をつけてrenameする。
          4桁連番+'_'をもつファイル名が存在する場合、その最大数の次から連番をつける
  --mode: "merge" (デフォルト)の場合、mergeを実行
  --filetype: "pdf", "docx", "pptx" (デフォルト)の場合、pdf

  -work_dir: ソースPDFファイルのディレクトリ
             マージしたファイルは merged サブディレクトリに保存される
  --nmerge: 1つのPDFファイルのマージする最大ファイル数
"""


def initialize():
    parser = argparse.ArgumentParser(description="PDF/Word/PowerPointファイルの連結またはPDFのリネームを行います。")
    parser.add_argument(
        "work_dir",
        nargs="?",
        default=".",
        help="対象ファイルを検索するディレクトリ。デフォルト: '.' (カレントディレクトリ)"
    )
    parser.add_argument(
        "-n",
        "--nmerge",
        type=int,
        default=10,
        help="連結するファイルの数（mergeモードのみ有効）。デフォルト: 10"
    )
    parser.add_argument(
        "-m",
        "--mode",
        choices=['merge', 'rename'],
        default="pdf_merge",
        help="実行モードを選択します: 'merge' (連結), 'rename' (ファイル名に連番をつけてリネーム)"
    )
    parser.add_argument(
        "-t",
        "--filetype",
        choices=['pdf', 'docx', 'pptx'],
        default="pdf",
        help="処理するファイルタイプを選択します: 'pdf', 'docx', 'pptx'"
    )
    parser.add_argument(
        '--max_depth',
        type=int,
        default=0,
        help='検索するディレクトリの深さを制限します (デフォルト: 0)'
    )
    return parser

def get_ms_gothic_path():
    """
    MS Gothicフォントのパスを取得します。
    """
    windows_dir = os.environ.get('SYSTEMROOT')
    if windows_dir:
        font_path = os.path.join(windows_dir, 'Fonts', 'msgothic.ttc')
        if os.path.exists(font_path):
            return font_path
    return None

def split_string_by_char_count(text, char_count):
    """
    文字列を指定された文字数ごとに分割し、リストで返します。
    """
    return [text[i:i+char_count] for i in range(0, len(text), char_count)]

def display_fixed_width_text():
    """
    30文字ごとに改行してPDFにテキストを表示する例
    """
    pdf = FPDF()
    pdf.add_page()

    # 日本語フォントを事前に登録する必要があります
    # 例: pdf.add_font("MS-Gothic", fname="C:/Windows/Fonts/msgothic.ttc")
    # pdf.set_font("MS-Gothic", size=12)
    
    pdf.set_font("Arial", size=12) # フォントが未登録の場合
    
    long_text = "このテキストは30文字ごとに自動で改行されるように分割されます。長い文章でも、この方法を使えば好きな文字数で改行できます。日本語のテキストも問題ありません。"
    
    # 30文字ごとに文字列を分割
    chunks = split_string_by_char_count(long_text, 30)

    # 分割された各文字列を1行ずつ表示
    for chunk in chunks:
        # cell()を使って1行ずつ表示し、ln=1で自動改行
        pdf.cell(0, 10, chunk, ln=1)

    pdf.output("fixed_wrap_example.pdf")
    print("PDFを生成しました: fixed_wrap_example.pdf")

def create_chunk_summary_page(filenames, fontsize = 12, nchars_split = 60):
    """
    チャンク全体のファイルリストをPDFページとして作成し、バッファとして返します。
    """
    pdf = FPDF()
    font_path = get_ms_gothic_path()

    if font_path:
        pdf.add_font("MS-Gothic", fname=font_path)
    else:
        print("警告: MS Gothicフォントが見つかりません。日本語は正しく表示されない可能性があります。", file=sys.stderr)
        
    pdf.add_page()
    
    if font_path:
        pdf.set_font("MS-Gothic", size=fontsize)
    else:
        pdf.set_font("Arial", size=fontsize)

    # タイトルを表示
    pdf.multi_cell(pdf.w - pdf.l_margin - pdf.r_margin, 10, "--- このチャンクで連結されるファイル ---")
    pdf.ln(10)
    
    for filename in filenames:
        base_filename = os.path.basename(filename)
        
        # ファイル名を分割
        chunks = split_string_by_char_count(base_filename, nchars_split)
        for chunk in chunks:
            pdf.cell(0, 10, chunk, ln=1)
        
        # 各ファイル名の間に空行を入れる
#        pdf.ln(5)

    pdf_buffer = io.BytesIO(pdf.output())
    pdf_buffer.seek(0)
    return pdf_buffer


def create_filename_page(filename, fontsize = 12, nchars_split = 60):
    """
    単一のファイル名を含むPDFページを作成し、バッファとして返します。
    """
    pdf = FPDF()
    font_path = get_ms_gothic_path()
    
    if font_path:
        pdf.add_font("MS-Gothic", fname=font_path)
    else:
        print("警告: MS Gothicフォントが見つかりません。日本語は正しく表示されない可能性があります。", file=sys.stderr)
        
    pdf.add_page()

    if font_path:
        pdf.set_font("MS-Gothic", size=fontsize)
    else:
        pdf.set_font("Arial", size=fontsize)
    
    page_height = pdf.h
    font_height = fontsize
    top_margin = (page_height - font_height) / 2
    
    pdf.ln(top_margin)
    
    cell_width = pdf.w - pdf.l_margin - pdf.r_margin
    
#    pdf.multi_cell(cell_width, 1, os.path.basename(filename), align=Align.C)
    filename = os.path.abspath(filename)
    # ファイル名を分割
    chunks = split_string_by_char_count(filename, nchars_split)
    for chunk in chunks:
        pdf.cell(0, 10, chunk, ln=1)
#        pdf.ln(5)

    pdf_buffer = io.BytesIO(pdf.output())
    pdf_buffer.seek(0)
    return pdf_buffer
        
def search_recursive(base_dir, pattern, max_depth):
    print("dir:", base_dir)
    print("pattern: ", pattern)
    print("max_depth:", max_depth)
    
    file_list = []
    for root, dirs, files in os.walk(base_dir):
        depth = len(root.split(os.sep)) - len(base_dir.split(os.sep))
        if depth >= max_depth + 1:
            dirs[:] = []  # 下位ディレクトリを無視
            continue

        for file in files:
            if file.replace('　', ' ') == pattern.replace('　', ' ') or fnmatch.fnmatch(file, pattern):
                print(f"Found matching file: {file}")
                file_list.append(os.path.join(root, file))
    return file_list
    
def merge_pdfs(work_dir, nmerge, max_depth):
    """
    指定されたディレクトリ内のPDFを、nmergeファイルごとに連結して'merged'ディレクトリに保存します。
    連結前に、チャンク全体のファイルリストページと、各PDFファイル名のページを追加します。
    """
    print("\n--- PDF連結モードを開始 ---")

    merged_dir = os.path.join(work_dir, "merged")
    
    if os.path.exists(merged_dir):
        existing_pdf_files_in_merged = glob.glob(os.path.join(merged_dir, "*.pdf"))
        if len(existing_pdf_files_in_merged) > 0:
            print(f"警告: '{merged_dir}' ディレクトリには既に {len(existing_pdf_files_in_merged)} 個のPDFファイルが存在します。")
            for f in existing_pdf_files_in_merged:
                print(f"  [{f}]")

            ret = input("\n  続行する場合は 'y' を入力してください >> ")
            if ret.lower() == 'y':
                pass
            else:
                print(f"  '{ret}' が入力されました。処理を終了します。")
                return False
        else:
            print(f"警告: '{merged_dir}' ディレクトリは存在しますが、PDFファイルはありません。続行します。")
    else:
        print(f"'{merged_dir}' ディレクトリを作成します。")
        os.makedirs(merged_dir, exist_ok=True)
    
#    pdf_files = sorted(glob.glob(os.path.join(work_dir, "*.pdf")))
    pdf_files = sorted(search_recursive(work_dir, "*.pdf", max_depth))
    
    if not pdf_files:
        print(f"警告: '{work_dir}' ディレクトリにPDFファイルが見つかりませんでした。")
        return

    chunks = [pdf_files[i:i + nmerge] for i in range(0, len(pdf_files), nmerge)]
    
    idx = 1
    for i, chunk in enumerate(chunks):
        print(f"\n--- チャンク {i+1} の処理を開始 ---")

        merger = PdfMerger()
        
        current_idx = idx
        while True:
            range_str = ""
            if chunk:
                numbers = []
                for f in chunk:
                    match = re.match(r'^(\d{4})_', os.path.basename(f))
                    if match:
                        numbers.append(int(match.group(1)))
                
                if numbers:
                    min_num = min(numbers)
                    max_num = max(numbers)
                    range_str = f"_{min_num:04d}-{max_num:04d}"
                else:
                    range_str = f"_chunk{i+1}"
            
            output_filename = f"merged_{current_idx:03d}{range_str}.pdf"
            output_path = os.path.join(merged_dir, output_filename)
            if not os.path.exists(output_path):
                break
            current_idx += 1
        idx = current_idx + 1
        
        print(f"  出力パス: [{output_path}]")

        if chunk:
            filenames_only_for_summary = [os.path.basename(f) for f in chunk]
            chunk_summary_pdf_buffer = create_chunk_summary_page(filenames_only_for_summary)
            merger.append(PdfReader(chunk_summary_pdf_buffer))
            print(f"  チャンク全体のファイルリストページを追加しました。")

        failed_files = []
        for pdf_file in chunk:
            print(f"  + 処理中: [{pdf_file}]")
            
            try:
                filename_page_buffer = create_filename_page(pdf_file)
                merger.append(PdfReader(filename_page_buffer))
                print(f"    ファイル名ページ '{os.path.basename(pdf_file)}' を追加しました。")
            except Exception as e:
                print(f"    警告: ファイル名ページ '{os.path.basename(pdf_file)}' の作成または追加に失敗しました。エラー: {e}")

            try:
                merger.append(pdf_file)
            except Exception as e:
                print(f"    失敗: '{pdf_file}' の読み込みに失敗しました。このファイルは連結対象外です。エラー: {e}")
                failed_files.append(pdf_file)
        
        if len(chunk) > len(failed_files) or (len(chunk) == 0 and len(failed_files) == 0):
            try:
                merger.write(output_path)
                print(f"  '{output_path}' を作成しました。")
            except Exception as e:
                print(f"  エラー: '{output_path}' の書き込みに失敗しました。エラー: {e}")
            finally:
                merger.close()
        else:
            print(f"  このチャンクでは連結可能なPDFがありませんでした。'{output_path}' は作成されません。")
        
        for failed_file in failed_files:
            try:
                shutil.copy(failed_file, merged_dir)
                print(f"  読み込みに失敗したファイル '{os.path.basename(failed_file)}' を'merged'ディレクトリにコピーしました。")
            except Exception as e:
                print(f"  エラー: ファイル '{failed_file}' のコピーに失敗しました。エラー: {e}")
        
        print(f"--- チャンク {i+1} の処理を終了 ---\n")

    return True

# --- Word連結関数 ---

def merge_word_docs(work_dir, nmerge):
    """
    指定されたディレクトリ内のWord文書を、nmergeファイルごとに連結して'merged'ディレクトリに保存します。
    注意: この機能は基本的なコンテンツの追加であり、複雑な書式設定やセクションの維持は保証されません。
    """
    if Document is None:
        print("エラー: 'python-docx' がインストールされていないため、Wordファイルの処理はできません。")
        return False

    print("\n--- Word文書連結モードを開始 ---")

    merged_dir = os.path.join(work_dir, "merged")
    os.makedirs(merged_dir, exist_ok=True)

    docx_files = sorted(glob.glob(os.path.join(work_dir, "*.docx")))

    if not docx_files:
        print(f"警告: '{work_dir}' ディレクトリにWordファイルが見つかりませんでした。")
        return

    chunks = [docx_files[i:i + nmerge] for i in range(0, len(docx_files), nmerge)]

    idx = 1
    for i, chunk in enumerate(chunks):
        print(f"\n--- チャンク {i+1} の処理を開始 ---")
        
        # 新しいWord文書を作成
        merged_doc = Document()

        # 出力ファイル名に連番範囲またはチャンク番号を追加
        range_str = ""
        if chunk:
            numbers = []
            for f in chunk:
                match = re.match(r'^(\d{4})_', os.path.basename(f))
                if match:
                    numbers.append(int(match.group(1)))
            
            if numbers:
                min_num = min(numbers)
                max_num = max(numbers)
                range_str = f"_{min_num:04d}-{max_num:04d}"
            else:
                range_str = f"_chunk{i+1}"
        
        current_idx = idx
        while True:
            output_filename = f"merged_word_{current_idx:03d}{range_str}.docx"
            output_path = os.path.join(merged_dir, output_filename)
            if not os.path.exists(output_path):
                break
            current_idx += 1
        idx = current_idx + 1

        print(f"  出力パス: [{output_path}]")

        failed_files = []
        for docx_file in chunk:
            print(f"  + 処理中: [{docx_file}]")
            try:
                source_doc = Document(docx_file)
                # ファイル名を追加する段落
                merged_doc.add_paragraph(f"--- {os.path.basename(docx_file)} の開始 ---").add_run().bold = True
                merged_doc.add_paragraph() # 空行

                for element in source_doc.element.body:
                    if element.tag.endswith('p'): # 段落の場合
                        merged_doc.add_paragraph(element.text)
                    elif element.tag.endswith('tbl'): # 表の場合
                        # 表をコピーする基本的な方法（複雑な表は完全には再現できない可能性あり）
                        # source_docから表を直接コピーする方法はないため、新しい表を作成し内容を移す
                        # これは非常に基本的なコピーであり、元の表の複雑な書式は失われる可能性があります
                        # 詳細な表のコピーにはもっと複雑なロジックが必要です
                        # ここでは、簡略化のため、表のテキストコンテンツのみをコピーします
                        merged_doc.add_paragraph(f"[表の内容: {os.path.basename(docx_file)} の表]")
                        for row in source_doc.tables[0].rows: # 最初の表だけを例として
                            row_text = [cell.text for cell in row.cells]
                            merged_doc.add_paragraph(" | ".join(row_text))
                merged_doc.add_page_break() # 各文書の終わりに改ページを追加
                print(f"    '{os.path.basename(docx_file)}' のコンテンツを連結しました。")
            except Exception as e:
                print(f"    失敗: '{docx_file}' の読み込みまたは連結に失敗しました。エラー: {e}")
                failed_files.append(docx_file)
        
        if len(chunk) > len(failed_files):
            try:
                merged_doc.save(output_path)
                print(f"  '{output_path}' を作成しました。")
            except Exception as e:
                print(f"  エラー: '{output_path}' の書き込みに失敗しました。エラー: {e}")
        else:
            print(f"  このチャンクでは連結可能なWord文書がありませんでした。'{output_path}' は作成されません。")
        
        for failed_file in failed_files:
            try:
                shutil.copy(failed_file, merged_dir)
                print(f"  読み込みに失敗したファイル '{os.path.basename(failed_file)}' を'merged'ディレクトリにコピーしました。")
            except Exception as e:
                print(f"  エラー: ファイル '{failed_file}' のコピーに失敗しました。エラー: {e}")
        
        print(f"--- チャンク {i+1} の処理を終了 ---\n")
    return True

# --- PowerPoint連結関数 ---

def merge_powerpoint_presentations(work_dir, nmerge):
    """
    指定されたディレクトリ内のPowerPointプレゼンテーションを、nmergeファイルごとに連結して'merged'ディレクトリに保存します。
    注意: スライドのレイアウトやマスターが結合先のプレゼンテーションと異なる場合、見た目が変わる可能性があります。
    """
    if Presentation is None:
        print("エラー: 'python-pptx' がインストールされていないため、PowerPointファイルの処理はできません。")
        return False

    print("\n--- PowerPointプレゼンテーション連結モードを開始 ---")

    merged_dir = os.path.join(work_dir, "merged")
    os.makedirs(merged_dir, exist_ok=True)

    pptx_files = sorted(glob.glob(os.path.join(work_dir, "*.pptx")))

    if not pptx_files:
        print(f"警告: '{work_dir}' ディレクトリにPowerPointファイルが見つかりませんでした。")
        return

    chunks = [pptx_files[i:i + nmerge] for i in range(0, len(pptx_files), nmerge)]

    idx = 1
    for i, chunk in enumerate(chunks):
        print(f"\n--- チャンク {i+1} の処理を開始 ---")
        
        if not chunk:
            print("  チャンクにPowerPointファイルがありません。スキップします。")
            continue

        # 最初のプレゼンテーションをベースとして開く
        try:
            base_prs = Presentation(chunk[0])
            print(f"  ベースプレゼンテーション: [{os.path.basename(chunk[0])}]")
        except Exception as e:
            print(f"  エラー: ベースプレゼンテーション '{chunk[0]}' の読み込みに失敗しました。このチャンクはスキップされます。エラー: {e}")
            continue

        # 出力ファイル名に連番範囲またはチャンク番号を追加
        range_str = ""
        if chunk:
            numbers = []
            for f in chunk:
                match = re.match(r'^(\d{4})_', os.path.basename(f))
                if match:
                    numbers.append(int(match.group(1)))
            
            if numbers:
                min_num = min(numbers)
                max_num = max(numbers)
                range_str = f"_{min_num:04d}-{max_num:04d}"
            else:
                range_str = f"_chunk{i+1}"
        
        current_idx = idx
        while True:
            output_filename = f"merged_pptx_{current_idx:03d}{range_str}.pptx"
            output_path = os.path.join(merged_dir, output_filename)
            if not os.path.exists(output_path):
                break
            current_idx += 1
        idx = current_idx + 1

        print(f"  出力パス: [{output_path}]")

        failed_files = []
        # ベース以外のプレゼンテーションからスライドをコピー
        for j, pptx_file in enumerate(chunk):
            if j == 0: # ベースプレゼンテーションはスキップ
                continue
            
            print(f"  + 処理中: [{pptx_file}]")
            try:
                source_prs = Presentation(pptx_file)
                # ファイル名を示すスライドを追加 (簡単なタイトルスライドとして)
                title_slide_layout = base_prs.slide_layouts[0] # 通常、最初のレイアウトはタイトルスライド
                new_slide = base_prs.slides.add_slide(title_slide_layout)
                title = new_slide.shapes.title
                title.text = f"--- {os.path.basename(pptx_file)} の開始 ---"
                print(f"    ファイル名スライド '{os.path.basename(pptx_file)}' を追加しました。")

                # ソースプレゼンテーションからスライドをコピー
                for slide in source_prs.slides:
                    # 新しいスライドを作成し、元のスライドのレイアウトを適用しようと試みる
                    # レイアウトが存在しない場合は、デフォルトのレイアウトを使用
                    try:
                        slide_layout = base_prs.slide_layouts[slide.slide_layout.name]
                    except KeyError:
                        slide_layout = base_prs.slide_layouts[0] # デフォルトのタイトルスライドレイアウト

                    new_slide = base_prs.slides.add_slide(slide_layout)

                    # シェイプをコピー
                    for shape in slide.shapes:
                        if shape.has_text_frame:
                            text_frame = new_slide.shapes.add_textbox(shape.left, shape.top, shape.width, shape.height).text_frame
                            text_frame.text = shape.text_frame.text
                            # フォントやサイズなどの詳細な書式設定はコピーされない可能性がある
                        elif shape.shape_type == MSO_SHAPE_TYPE.PICTURE:
                            # 画像のコピーは複雑なため、ここではプレースホルダーとして扱う
                            # 画像データを直接コピーするには、より高度な処理が必要
                            # 例: new_slide.shapes.add_picture(image_stream, left, top, width, height)
                            # image_stream を取得するには、元の画像データを読み込む必要がある
                            new_slide.shapes.add_textbox(shape.left, shape.top, shape.width, shape.height).text_frame.text = "[画像]"
                        elif shape.has_table:
                            # 表のコピーも複雑なため、ここではテキストとして扱う
                            new_slide.shapes.add_textbox(shape.left, shape.top, shape.width, shape.height).text_frame.text = "[表]"
                        else:
                            # その他のシェイプはコピーしないか、基本的なシェイプとして追加
                            pass # ここでは何もしない
                    
                    print(f"    スライドをコピーしました。")

            except Exception as e:
                print(f"    失敗: '{pptx_file}' の読み込みまたは連結に失敗しました。エラー: {e}")
                failed_files.append(pptx_file)
        
        if len(chunk) > len(failed_files):
            try:
                base_prs.save(output_path)
                print(f"  '{output_path}' を作成しました。")
            except Exception as e:
                print(f"  エラー: '{output_path}' の書き込みに失敗しました。エラー: {e}")
        else:
            print(f"  このチャンクでは連結可能なPowerPointプレゼンテーションがありませんでした。'{output_path}' は作成されません。")
        
        for failed_file in failed_files:
            try:
                shutil.copy(failed_file, merged_dir)
                print(f"  読み込みに失敗したファイル '{os.path.basename(failed_file)}' を'merged'ディレクトリにコピーしました。")
            except Exception as e:
                print(f"  エラー: ファイル '{failed_file}' のコピーに失敗しました。エラー: {e}")
        
        print(f"--- チャンク {i+1} の処理を終了 ---\n")
    return True

# --- PDFリネーム関数 ---

def rename_files(work_dir, filetype = 'pdf'):
    """
    指定されたディレクトリ内のファイルをリネームします。
    ファイル名が数字4桁+'_'で始まるものから最大の数字を抽出し、
    それ以外のファイルをソートして、その次の連番から4桁+'_'を付けてリネームします。
    """

    print(f"\nStart {filetype.upper()} rename")

    pdf_files = sorted(glob.glob(os.path.join(work_dir, f"*.{filetype}")))
    
    if not pdf_files:
        print(f"警告: '{work_dir}' ディレクトリに{filetype.upper()}ファイルが見つかりませんでした。")
        return

    max_idx = -1
    unprefixed_files = []

    for f in pdf_files:
        basename = os.path.basename(f)
        match = re.match(r'^(\d{4})_', basename)
        if match:
            current_idx = int(match.group(1))
            if current_idx > max_idx:
                max_idx = current_idx
            print(f"  既存の連番ファイル: [{basename}] (インデックス: {current_idx})")
        else:
            unprefixed_files.append(f)
            print(f"  プレフィックスなしファイル: [{basename}]")

    start_idx = max_idx + 1
    if start_idx == 0:
        start_idx = 1
    
    print(f"\n  リネーム開始インデックス: {start_idx:04d}")

    for i, old_path in enumerate(sorted(unprefixed_files)):
        basename = os.path.basename(old_path)
        new_prefix = f"{start_idx + i:04d}_"
        new_basename = new_prefix + basename
        new_path = os.path.join(work_dir, new_basename)

        if os.path.exists(new_path):
            print(f"  警告: リネーム先のファイル '{new_path}' が既に存在します。スキップします。")
            continue
        
        try:
            os.rename(old_path, new_path)
            print(f"  リネーム: '{basename}' -> '{new_basename}'")
        except Exception as e:
            print(f"  エラー: '{old_path}' のリネームに失敗しました。エラー: {e}")
    
    print(f"\nFinish {filetype.upper()} rename")
    return True

def convert(mode, work_dir, filetype, nmerge, max_depth):
    if nmerge <= 0:
        nmerge = 100000000

    if not os.path.isdir(work_dir):
        print(f"エラー: 指定されたディレクトリ '{work_dir}' が見つかりません。")
        return False

    if mode == "merge" and filetype == 'pdf':
        merge_pdfs(work_dir, nmerge, max_depth)
    elif mode == "pdf_rename":
       rename_files(work_dir, filetype)
    elif mode == "merge" and filetype == 'docx':
        merge_word_docs(work_dir, nmerge)
    elif mode == "word_rename":
        rename_files(work_dir, filetype)
    elif mode == "merge" and filetype == 'pptx':
        merge_powerpoint_presentations(work_dir, nmerge)
    elif mode == "powerpoint_rename":
        rename_files(work_dir, filetype)

    return True


if __name__ == "__main__":
    parser = initialize()
    args = parser.parse_args()
    convert(args.mode, args.work_dir, args.filetype, args.nmerge, args.max_depth)

    input("\nPress Enter to terminate>>\n")


