import os
import sys
import fnmatch
import argparse
import glob
import convert2md
import txt2pdf
import pptx2pdf
import pptx2img
import pptx2md
import docx2img
import pdf2img
import pdf2md
import img2pdf
try:
    import html2pdf
except:
    html2pdf = None


# 引数の設定とパースを行う関数
def initialize():
    """
    コマンドライン引数を解析し、ArgumentParserオブジェクトを返す
    """
    # ヘルプメッセージ付きのパーサーを作成
    parser = argparse.ArgumentParser(
        description='PowerPointファイルを画像に変換します。',
        formatter_class=argparse.RawTextHelpFormatter
    )

    # 引数を追加
    parser.add_argument(
        '--infile',
        default='41-01-search_MP.pptx',
        help='入力ファイル(pptx/docx/pdf)のパスを指定します。\n'
             '(デフォルト: 41-01-search_MP.pptx)'
    )
    
    parser.add_argument(
        '--output_dir',
        default='images',
        help='画像を保存するディレクトリのパスを指定します。\n'
             '(デフォルト: images)'
    )
    
    parser.add_argument(
        '--output_format',
        default='png',
        choices=['png', 'jpg', 'jpeg', 'md', 'pdf'],
        help='出力画像のフォーマットを指定します。\n'
             '現在、png/md のみがサポートされています。\n'
             '(デフォルト: png)'
    )

    parser.add_argument(
        '--max_depth',
        type=int,
        default=0,
        help='検索するディレクトリの深さを制限します (デフォルト: 0)'
    )

    # 引数を解析して返す
    return parser

def convert(infile, output_dir, output_format):
    infile_ext = os.path.splitext(infile)[1].lower()
    print()
    print(f"入力ファイル: {infile} (拡張子: {infile_ext})")
    print(f"出力ディレクトリ: {output_dir}")
    print(f"出力フォーマット: {output_format}")
    print(f"入力ファイルタイプ: {infile_ext}")

    if os.path.getsize(infile) == 0:
        print(f"Error: 入力ファイル '{infile}' は空です。", file=sys.stderr)
        return False

    infile_type = infile_ext.replace('.', '').lower()
    if infile_type == output_format.lower():
        print(f"Input type [{infile_type}] is the same as the output_format [{output_format}]. Skip.")
        return True
    elif infile_ext.lower() in ['png', 'jpg', 'jpeg']:
        if output_format == 'pdf':
            img2pdf.convert_image_to_pdf(infile)
            return False
    elif infile_ext == ".pptx":
        if output_format in ['png', 'jpg', 'jpeg']:
            pptx2img.export_img(infile, output_dir, output_format)
            pptx2img.rename_img(output_dir) # out_dirはoutput_dirに変更
        elif output_format == 'pdf':
            filename_without_ext = os.path.splitext(os.path.basename(infile))[0]
            pdf_path = os.path.join(os.path.dirname(infile), f"{filename_without_ext}.pdf")
            pptx2pdf.pptx_to_pdf(infile, pdf_path)
        elif output_format == 'md':
#            pptx2md.convert(infile, None)
            convert2md.convert(infile, None)
        else:
            print(f"\nError: Invalid output_format [{output_format}] for input type [{infile_ext}]\n")
            return False
    elif infile_ext == ".docx":
        if output_format in ['png', 'jpg', 'jpeg']:
            filename_without_ext = os.path.splitext(os.path.basename(infile))[0]
            pdf_path = os.path.join(os.path.dirname(infile), f"{filename_without_ext}.pdf")
            docx2img.convert_word_to_pdf(infile, pdf_path)
            docx2img.convert_pdf_to_images(pdf_path, output_dir)
        elif output_format == 'pdf':
            filename_without_ext = os.path.splitext(os.path.basename(infile))[0]
            pdf_path = os.path.join(os.path.dirname(infile), f"{filename_without_ext}.pdf")
            docx2img.convert_word_to_pdf(infile, pdf_path)
        elif output_format == 'md':
            convert2md.convert(infile, None)
        else:
            print(f"\nError: Invalid output_format [{output_format}] for input type [{infile_ext}]\n")
            return False
    elif infile_ext == ".pdf":
        if output_format in ['png', 'jpg', 'jpeg']:
            pdf2img.convert_pdf_to_images(infile, output_dir, output_format)
        elif output_format == 'md':
            pdf2md.convert(infile, None)
        elif output_format == 'md':
            convert2md.convert(infile, None)
        else:
            print(f"\nError: Invalid output_format [{output_format}] for input type [{infile_ext}]\n")
            return False
    elif infile_ext == ".txt" or infile_ext == ".md":
        if output_format == 'pdf':
            txt2pdf.convert_to_pdf(infile, None)
        else:
            print(f"\nError: Invalid output_format [{output_format}] for input type [{infile_ext}]\n")
            return False
    elif html2pdf and (infile_ext == ".html" or infile_ext == ".htm"):
        if output_format == 'pdf':
            html2pdf.convert_to_pdf(infile, None)
        else:
            print(f"\nError: Invalid output_format [{output_format}] for input type [{infile_ext}]\n")
            return False

    return True

def main():
    parser = initialize()
    args = parser.parse_args()

    pattern = os.path.basename(args.infile)
    base_dir = os.path.dirname(os.path.abspath(args.infile))

    print()
    print(f"入力ファイル: {args.infile}")
    print(f"入力ファイルのパターン: {pattern}")
    print(f"入力ファイルのディレクトリ: {base_dir}")
    print(f"出力フォーマット: {args.output_format}")
    print(f"出力ディレクトリ: {args.output_dir}")
    file_list = []
    for root, dirs, files in os.walk(base_dir):
        depth = len(root.split(os.sep)) - len(base_dir.split(os.sep))
        if depth >= args.max_depth + 1:
            dirs[:] = []  # 下位ディレクトリを無視
            continue

        for file in files:
            if file.replace('　', ' ') == pattern.replace('　', ' ') or fnmatch.fnmatch(file, pattern):
                print(f"Found matching file: {file}")
                file_list.append(os.path.join(root, file))

    print("Files found:")
    for f in file_list:
        print(f"   {f}")

    for f in file_list:
        convert(f, args.output_dir, args.output_format)
    input("\n終了するにはENTERキーを押してください>>\n")


if __name__ == '__main__':
    main()