以下は、`binparser.py` ライブラリの解析結果です。

---

## `binparser.py` ライブラリ解析

### 1) このライブラリの主な機能や目的

`binparser.py` は、バイナリファイルを読み込み、その内容を特定のデータ型（整数、浮動小数点数、ASCII文字列）として解釈したり、ファイル内で特定の値を検索したりするためのPythonライブラリです。

主な機能は以下の通りです。

*   **バイナリファイルの読み込み**: ファイル全体をバイト列としてメモリに読み込みます。
*   **データ抽出**: 指定されたオフセット、サイズ、データ型、エンディアンに基づいて、バイナリデータから単一の値を抽出します。
*   **データ検索**: バイナリデータ全体または指定範囲内で、特定の数値（浮動小数点数には許容誤差も指定可能）や文字列を検索し、そのオフセットを特定します。
*   **型マッピング**: `struct` モジュールで使用される型フォーマットと、各データ型が占めるバイトサイズを内部的に管理しています。

このライブラリの目的は、バイナリデータ解析（例えば、カスタムファイルフォーマットの解析、組み込みシステムのダンプ解析など）を容易にすることにあります。

### 2) このライブラリを他のプログラムからimportする方法

`binparser.py` をPythonパスが通っているディレクトリに配置するか、実行中のスクリプトと同じディレクトリに配置することで、以下の方法でインポートできます。

```python
# ライブラリ全体をインポートし、binparser.pyの関数を 'binparser.' プレフィックスで呼び出す
import binparser

binary_data = binparser.load_bin_file("my_data.bin")
value = binparser.get_data(binary_data, 0x10, 4, 'float32', 'little')
offsets = binparser.search_data(binary_data, 0, len(binary_data), 4, 'int32', 'big', 12345)

# 特定の関数のみをインポートし、直接呼び出す
from binparser import load_bin_file, get_data, search_data

binary_data = load_bin_file("my_data.bin")
value = get_data(binary_data, 0x10, 4, 'float32', 'little')
offsets = search_data(binary_data, 0, len(binary_data), 4, 'int32', 'big', 12345)
```

### 3) 必要な非標準ライブラリとインストールコマンドとインストール方法

`binparser.py` は、以下の標準ライブラリのみを使用しており、**非標準ライブラリは使用していません**。

*   `sys`
*   `struct`
*   `re`

したがって、**追加のインストールは不要です**。Pythonがインストールされていれば、そのまま使用できます。

### 4) importできる変数と関数

`__all__` 変数で公開されているのは `load_bin_file`, `get_data`, `search_data` の3つの関数ですが、`type_map` 変数も実質的に利用可能なグローバル変数です。

#### 変数

*   `type_map`:
    ```python
    type_map = {
            'float32': ('f', 4),
            'float64': ('d', 8),
            'uint16':  ('H', 2),
            'int16':   ('h', 2),
            'uint32':  ('I', 4),
            'int32':   ('i', 4),
            'uint64':  ('Q', 8),
            'int64':   ('q', 8),
        }
    ```
    *   **説明**: Pythonの `struct` モジュールで使用されるフォーマット文字列と、それに対応するバイトサイズをマッピングする辞書です。キーはデータ型名（例: `'float32'`, `'uint16'`）、値はタプル `(format_char, byte_size)` です。`format_char` は `struct` モジュールがバイトデータを解釈する際に使用する文字（例: `'f'` for float32, `'H'` for unsigned short）、`byte_size` はその型が占めるバイト数です。

#### 関数

*   `load_bin_file(file_path)`
    *   **動作**: 指定されたパスのバイナリファイルを読み込み、その内容をバイト列として返します。
    *   **引数**:
        *   `file_path` (str): 読み込むバイナリファイルのパス。
    *   **戻り値**:
        *   `bytes`: ファイルの内容全体を含むバイト列。

*   `get_data(bin_data, offset, size, vartype, endian='little', varname=None)`
    *   **動作**: 指定されたバイナリデータから、特定のオフセット、サイズ、データ型、エンディアンに基づいて単一の値を抽出します。文字列型の場合、指定された範囲の先頭からASCII文字を抽出し、非ASCII文字に遭遇するとそこで停止します。
    *   **引数**:
        *   `bin_data` (bytes): データ抽出の元となるバイナリデータ。
        *   `offset` (int): データを読み取り開始するオフセット（バイト単位）。
        *   `size` (int): 読み取るデータのバイトサイズ。
        *   `vartype` (str): 読み取るデータの型（例: `'float32'`, `'int16'`, `'str'` など。`type_map` に定義されている型、または `'str'`）。
        *   `endian` (str, 省略可能): エンディアン指定。`'little'` (リトルエンディアン) または `'big'` (ビッグエンディアン)。デフォルトは `'little'`。
        *   `varname` (str, 省略可能): 読み取りエラー発生時に表示される変数名。デバッグ情報として利用されます。
    *   **戻り値**:
        *   `int`, `float`, `str`, `None`: 抽出された値。型が数値型で読み取りに失敗した場合は `None` を返します。
    *   **例外**:
        *   `ValueError`: 未対応のエンディアン指定、または指定されたサイズが型に対して不正な場合、未対応の型が指定された場合に発生します。

*   `search_data(bin_data, itop, ilast, size, vartype, endian, value, tolerance=1e-6, varname='not given')`
    *   **動作**: 指定されたバイナリデータの範囲（`itop` から `ilast` まで）内で、特定の `value` を検索します。数値型の場合、`tolerance` を指定することで浮動小数点数の近似検索が可能です。文字列型の場合、指定された`value`がデコードされたバイト列内に含まれるかを検索します。
    *   **引数**:
        *   `bin_data` (bytes): 検索対象のバイナリデータ。
        *   `itop` (int): 検索を開始するオフセット（バイト単位）。
        *   `ilast` (int): 検索を終了するオフセット（バイト単位）。このオフセットは含まれます。
        *   `size` (int): 検索対象となるデータのバイトサイズ。
        *   `vartype` (str): 検索するデータの型（例: `'float32'`, `'int16'`, `'str'` など。`type_map` に定義されている型、または `'str'`）。
        *   `endian` (str): エンディアン指定。`'little'` (リトルエンディアン) または `'big'` (ビッグエンディアン)。
        *   `value` (int, float, str): 検索する値。
        *   `tolerance` (float, 省略可能): `vartype` が浮動小数点数の場合に許容する誤差。`abs(found_value - value) <= tolerance` で比較されます。デフォルトは `1e-6`。
        *   `varname` (str, 省略可能): デバッグ情報として利用される変数名。
    *   **戻り値**:
        *   `list[int]`: `value` が見つかったすべてのオフセットのリスト。見つからない場合は空のリストを返します。
    *   **例外**:
        *   `ValueError`: 未対応のエンディアン指定、または指定されたサイズが型に対して不正な場合、未対応の型が指定された場合に発生します。

---

### 5) main scriptとして実行したときの動作

`binparser.py` が直接実行された場合（`python binparser.py` のように）、`main()` 関数が呼び出されます。

1.  **引数解析**:
    *   デフォルトでは `bin_file` を `'test.raw'`、`endian` を `'little'` に設定します。
    *   コマンドライン引数が与えられた場合、最初の引数を `bin_file`（ファイルパス）として、2番目の引数を `endian`（`'little'`または`'big'`）として使用します。
        *   例: `python binparser.py mydata.bin big`

2.  **ファイル読み込み**:
    *   指定された`bin_file` を `load_bin_file()` 関数を使ってバイナリデータとしてメモリに読み込みます。

3.  **データ検索と結果出力**:
    *   読み込んだバイナリデータ (`bin_data`) に対して、`search_data()` 関数を用いて2種類の検索を実行します。
    *   **浮動小数点数 (`float32`) の検索**:
        *   以下の値 (`8.0`, `120.0`, `0.02`, `514.902`, `552.835`) を、バイトサイズ `4`、型 `'float32'`、指定された `endian`、許容誤差 `0.002` で検索します。
        *   各値について、見つかったすべてのオフセット（16進数形式）を標準出力に表示します。見つからない場合は「No data found for [値]」と表示します。
    *   **整数 (`int16`) の検索**:
        *   値 `5601` を、バイトサイズ `2`、型 `'int16'`、指定された `endian` で検索します。
        *   見つかったすべてのオフセット（16進数形式）を標準出力に表示します。見つからない場合は「No data found for [値]」と表示します。

この動作は、ライブラリの `search_data` 関数の使用例を示すデモンストレーションとして機能します。実行には `test.raw` というファイルが存在し、検索対象のデータが含まれていることを想定しています。