#pip install google-cloud-speech pydub google-generativeai python-dotenv openai google-api-core

import os
import io
from pydub import AudioSegment # pydubは音声ファイルの読み込みと変換に便利です
from google.cloud import speech_v1p1beta1 as speech # より新しいバージョンをインポート
#from google.cloud import storage # storageは不要になりますが、念のため残しておきます
import google.generativeai as genai
from dotenv import load_dotenv # dotenvのインポートを追加
from openai import OpenAI # OpenAIクライアントはGeminiと関係ないので、今回の要件では使用しませんが、元のコードに合わせて残します

# --- 設定 ---
config_path = "translate.env"
LOCAL_AUDIO_FILE = "generative_ai_programming.wav" # 処理したいローカル音声ファイル

# 環境変数読み込み
load_dotenv(dotenv_path=config_path)
account_inf_path = os.getenv("account_inf_path", "accounts.env")
load_dotenv(dotenv_path=account_inf_path)

# OpenAI APIキー設定 (このプログラムでは直接使用しませんが、元の構造を維持)
openai_api_key = os.getenv("OPENAI_API_KEY")
client = OpenAI(api_key=openai_api_key)

# Gemini APIキー設定
GEMINI_API_KEY = os.getenv("GEMINI_API_KEY") # 環境変数から取得

# --- 初期化 ---
# Google Cloud Storage クライアント (このプログラムでは使用しません)
# storage_client = storage.Client()

# Google Cloud Speech クライアント
# speech_v1p1beta1 は推奨されるAPIバージョンです。
speech_client = speech.SpeechClient()

# Gemini API の設定
if GEMINI_API_KEY:
    genai.configure(api_key=GEMINI_API_KEY)
else:
    raise ValueError("GEMINI_API_KEYが設定されていません。")

# --- 関数定義 ---

def transcribe_audio_from_memory(local_file_path):
    """
    ローカルの音声ファイルを読み込み、Google Cloud Speech-to-Text APIに直接送信して文字起こしします。
    短い音声ファイル（約1分未満）に適しています。
    """
    print(f"'{local_file_path}' をメモリから文字起こしを開始...")

    # 音声ファイルを読み込み、APIが受け付ける形式に変換 (必要に応じて)
    # ここでは、wavファイルを例にしていますが、MP3などの場合はpydubでwavに変換すると確実です
    try:
        audio_segment = AudioSegment.from_file(local_file_path)
        # APIの推奨設定に合わせる (例: 16kHz, mono)
        audio_segment = audio_segment.set_frame_rate(16000).set_channels(1)
        
        # バイナリデータとしてメモリに書き出す
        with io.BytesIO() as byte_io:
            audio_segment.export(byte_io, format="LINEAR16") # LINEAR16は生のPCMデータ
            content = byte_io.getvalue()

    except Exception as e:
        print(f"音声ファイルの読み込みまたは変換に失敗しました: {e}")
        print("pydubがffmpeg/libavを必要とする場合があります。インストールされているか確認してください。")
        return ""

    audio = speech.RecognitionAudio(content=content)
    
    # RecognitionConfigを設定
    # encoding: LINEAR16は生のPCMデータ、MP3は元のエンコーディング
    # 音声ファイルの実際のエンコーディングと一致させるか、上記でLINEAR16に変換してください。
    config = speech.RecognitionConfig(
        encoding=speech.RecognitionConfig.AudioEncoding.LINEAR16, # ここを変換後の形式に合わせる
        sample_rate_hertz=16000, # 変換後のサンプルレートに合わせる
        language_code="ja-JP", # 言語コード
    )

    # 同期処理で文字起こしを実行
    try:
        response = speech_client.recognize(config=config, audio=audio)
    except Exception as e:
        print(f"Speech-to-Text API呼び出しでエラーが発生しました: {e}")
        return ""

    transcript = ""
    for result in response.results:
        transcript += result.alternatives[0].transcript
    print("文字起こし完了。")
    return transcript

def summarize_text_with_gemini(text_to_summarize):
    """
    Gemini APIを使用してテキストを要約します。
    """
    print("Gemini APIで要約を開始...")
    model = genai.GenerativeModel('gemini-pro')
    
    prompt = f"""以下のテキストを要約してください。重要なポイントを抽出し、簡潔にまとめてください。

テキスト:
{text_to_summarize}

要約:
"""
    
    try:
        response = model.generate_content(prompt)
        # 応答が正常かどうかのチェックを強化
        if response and response.parts:
            summary = ""
            for part in response.parts:
                if hasattr(part, "text"):
                    summary += part.text
            print("要約完了。")
            return summary
        else:
            print("Gemini APIからの応答が空でした。")
            return "要約できませんでした: 応答が空でした。"
            
    except genai.types.BlockedPromptException as e:
        print(f"Gemini APIでのエラー: プロンプトがブロックされました: {e}")
        return f"要約できませんでした: プロンプトがブロックされました。"
    except genai.types.BrokenResponseException as e:
        print(f"Gemini APIでのエラー: 不完全な応答が返されました: {e}")
        return f"要約できませんでした: 不完全な応答。"
    except Exception as e: # より汎用的な例外ハンドリング
        print(f"Gemini APIでのエラー: {e}")
        return f"要約できませんでした: {e}"

# --- メイン処理 ---
if __name__ == "__main__":
    if not os.path.exists(LOCAL_AUDIO_FILE):
        print(f"エラー: ローカル音声ファイル '{LOCAL_AUDIO_FILE}' が見つかりません。パスを確認してください。")
        print("サンプル音声ファイルを作成する場合は、以下のようにできます:")
        print("  例: ffmpeg -f lavfi -i 'sine=frequency=1000:duration=3' -filter_complex 'amerge' test_audio.mp3")
        print("または、`pydub` を使って簡単な音声を作成することもできますが、`ffmpeg` が必要になる場合があります。")
        exit()

    try:
        # 1. 音声ファイルを直接文字起こし
        transcribed_text = transcribe_audio_from_memory(LOCAL_AUDIO_FILE)
        print("\n--- 文字起こし結果 ---")
        print(transcribed_text)

        if transcribed_text:
            # 2. 要約
            summarized_text = summarize_text_with_gemini(transcribed_text)
            print("\n--- 要約結果 (Gemini API) ---")
            print(summarized_text)
        else:
            print("\n文字起こしされたテキストがないため、要約できませんでした。")

    except Exception as e:
        print(f"\nエラーが発生しました: {e}")
        print("設定（APIキー、サービスアカウントキー）とファイルパスを確認してください。")
        print("Speech-to-Text APIとGenerative Language APIがGCPプロジェクトで有効になっていることを確認してください。")
        
        
        