#Tesseract OCR for Windows: https://tesseract-ocr.github.io/tessdoc/Downloads.html
# pip install Pillow PyMuPDF pytesseract

import os
import sys
import fitz  # PyMuPDF
from PIL import Image
import pytesseract


pdf_file = "peak_fit.pdf"

def ocr_from_pdf(pdf_path, lang='jpn', output_dir='pdf_images'):
    """
    PDFファイルから文字認識を行い、抽出されたテキストを返します。
    各ページは一時的に画像として保存され、OCR後に削除されます。

    Args:
        pdf_path (str): PDFファイルのパス。
        lang (str): 認識する言語。'jpn' (日本語) や 'eng' (英語) など。
        output_dir (str): PDFから抽出した一時画像を保存するディレクトリ。

    Returns:
        str: 抽出された全てのテキストを連結したもの。
    """
    full_text = []
    
    if not os.path.exists(output_dir):
        os.makedirs(output_dir)

    try:
        doc = fitz.open(pdf_path)
        print(f"PDFファイル '{pdf_path}' のページ数: {len(doc)}")

        for page_num in range(len(doc)):
            page = doc.load_page(page_num)
            
            # ページを画像としてレンダリング (高解像度で)
            # dpiを高く設定することで、より鮮明な画像になりOCR精度が向上する可能性があります
            pix = page.get_pixmap(matrix=fitz.Matrix(300/72, 300/72)) # 300 DPI
            
            image_path = os.path.join(output_dir, f"page_{page_num+1}.png")
            pix.save(image_path)
            print(f"  ページ {page_num+1} を '{image_path}' として保存しました。")

            # 画像からOCRを実行
            text = pytesseract.image_to_string(Image.open(image_path), lang=lang)
            full_text.append(f"\n--- ページ {page_num+1} のOCR結果 ---\n{text}")

            # 一時画像を削除
            os.remove(image_path)
            print(f"  一時画像 '{image_path}' を削除しました。")

        doc.close()
        return "\n".join(full_text)

    except FileNotFoundError:
        return f"エラー: ファイル '{pdf_path}' が見つかりません。"
    except Exception as e:
        return f"PDF OCR処理中にエラーが発生しました: {e}"
    finally:
        # 処理が中断されてもディレクトリが残らないように最終的にディレクトリを削除
        if os.path.exists(output_dir) and not os.listdir(output_dir):
            os.rmdir(output_dir)

if __name__ == "__main__":
    # テスト用のダミーPDFファイルを作成 (実際には既存のPDFを使用してください)
    if not os.path.exists(pdf_file):
        try:
        # PyMuPDFとPillowを使って簡単なダミーPDFを生成
            doc = fitz.open()  # 新しいPDFドキュメントを作成
            page = doc.new_page() # 新しいページを追加
        
        # テキストを追加
            page.insert_text((50, 50), "これはテストPDFです。", fontsize=20, fontname="Helvetica")
            page.insert_text((50, 80), "PDF OCRの例。", fontsize=15, fontname="Helvetica")
        
        # 日本語テキストを追加する場合は、適切なフォントとエンコーディングが必要になる場合があります
        # ここでは基本的な例としてASCII文字のみにしています。
        # 日本語を含むPDFを生成するには、より複雑な処理が必要です。

            doc.save(pdf_file)
            doc.close()
            print(f"ダミーPDF '{pdf_file}' を作成しました。")
        except Exception as e:
            print(f"ダミーPDFの作成に失敗しました: {e}")
            print("既存のPDFファイルを使用するか、手動でPDFファイルを作成してください。")
    

    print(f"\n--- PDFファイル '{pdf_file}' のOCR結果 ---")
    extracted_text_pdf = ocr_from_pdf(pdf_file, lang='jpn+eng')
    print(extracted_text_pdf)