import os
import sys
from types import SimpleNamespace
import argparse
import numpy as np
import pandas as pd
import matplotlib.pyplot as plt
import plotly.graph_objects as go  # plotly を使用する場合

def initialize():
    """ cfgオブジェクト作成および設定ファイルの読み込み """
    cfg = SimpleNamespace()

    parser = argparse.ArgumentParser(description="Scientific Computing Program")
    parser.add_argument("--mode", type=str, required=False, default="exec", choices=["exec"], help="Execution mode")
    parser.add_argument("--plot_mode", type=str, required=False, default="matplotlib", choices=["matplotlib", "plotly"], help="Plot mode")
    parser.add_argument("--infile", type=str, required=True, help="Input Excel data file")
    parser.add_argument("--fontsize", type=int, required=False, default=16, help="Graph font size")
    parser.add_argument("--fontsize_legend", type=int, required=False, default=12, help="Graph legend font size")
    parser.add_argument("--font_graph", type=str, required=False, default="MS Gothic", help="Graph font")
    parser.add_argument("--outfile", type=str, required=False, default=None, help="Output Excel file")

    return cfg, parser

def usage(parser):
    """ `argparse` を使って動的に使用方法を表示する """
    print("\n=== 使用方法 ===")
    parser.print_help()  # 引数の説明を自動表示

def update_vars(cfg, parser):
    """ コマンドライン引数の値を cfg にコピーする """
    try:
        args = parser.parse_args()
        for key, value in vars(args).items():
            setattr(cfg, key, value)
    except argparse.ArgumentError:
        usage(parser)
        sys.exit(1)

    # outfileが指定されていない場合、infileを基に生成
    if not getattr(cfg, "outfile", None):
       infile_basename = os.path.splitext(cfg.infile)[0]  # 拡張子を除去
       cfg.outfile = f"{infile_basename}.xlsx"

    return cfg

def read_data(filename):
    """ 
    Excelファイルからデータを読み込む関数 
    - filename: 読み込むExcelファイルのパス
    - return: 列ラベル (labels) とデータリスト (data_list)
    """
    try:
        df = pd.read_excel(filename)  # Excelファイルを pandas で読み込む
        labels = df.columns.tolist()
        data_list = df.to_numpy()
        return labels, data_list
    except Exception as e:
        print(f"データの読み込みに失敗しました: {e}")
        sys.exit(1)

def split_data(labels, data_list):
    """ データの前処理・分割 """
    xdata = data_list[:, 0]
    ydata = data_list[:, 1]
    return labels[0], labels[1], xdata, ydata

def analyze(xlabels, ylabels, xdata_list, ydata_list):
    """ 解析処理（仮の実装） """
    xcal_list = np.mean(xdata_list)
    ycal_list = np.mean(ydata_list)
    return xlabels, ylabels, xcal_list, ycal_list

def save_data(xlabels, ylabels, xdata_list, ydata_list, xlabels_cal, ylabels_cal, xcal_list, ycal_list):
    """ データ保存（仮の実装） """
    try:
        # フィッティング結果をデータフレームとして作成
        data = {
            "Coefficient": ["a", "b", "c"],
            "Value": coefficients
        }
        df = pd.DataFrame(data)
        
        # Excelファイルに保存
        df.to_excel(outfile, index=False)
        print(f"フィッティング結果を '{outfile}' に保存しました。")
    except Exception as e:
        print(f"結果の保存に失敗しました: {e}")
        sys.exit(1)

def plot_by_matplotlib(xlabels, ylabels, xdata_list, ydata_list, xlabels_cal, ylabels_cal, xcal_list, ycal_list):
    plt.rcParams["font.family"] = cfg.font_graph
    plt.xticks(fontsize=cfg.fontsize)  # X軸目盛りのフォントサイズ
    plt.yticks(fontsize=cfg.fontsize)  # Y軸目盛りのフォントサイズ
    plt.scatter(xdata_list, ydata_list, label="Original Data", color="blue")
    plt.axvline(x=xcal_list, color='r', linestyle="--", label=f"{xlabels} Mean")
    plt.axhline(y=ycal_list, color='b', linestyle="--", label=f"{ylabels} Mean")
    plt.xlabel(xlabels, fontsize = cfg.fontsize)
    plt.ylabel(ylabels, fontsize = cfg.fontsize)
    plt.legend(fontsize=cfg.fontsize_legend)
    plt.title("Least Squares Quadratic Fit", fontsize=cfg.fontsize)
    plt.tight_layout()
    plt.show()

def plot_by_plotly(xlabels, ylabels, xdata_list, ydata_list, xlabels_cal, ylabels_cal, xcal_list, ycal_list):
    fig = go.Figure()
    fig.add_trace(go.Scatter(x=xdata, y=ydata, mode="markers", name="Original Data", marker=dict(color="blue")))
    fig.add_trace(go.Scatter(x=x_fit, y=y_fit, mode="lines", name="Quadratic Fit", line=dict(color="red")))
    fig.update_layout(
            plot_bgcolor="white",  # プロット枠背景を白に設定
            xaxis=dict(
                title="X",
                tickfont=dict(size=cfg.fontsize),
                showline=True,  # 枠線を表示
                linewidth=2,    # 枠線の太さを設定
                linecolor="black",  # 枠線の色を黒に設定
                ticks="inside",  # 目盛り線を内側に表示
                mirror=True,
                ticklen=10,
                tickwidth=2,
            ),
            yaxis=dict(
                title="Y",
                tickfont=dict(size=cfg.fontsize),
                showline=True,  # 枠線を表示
                linewidth=2,    # 枠線の太さを設定
                linecolor="black",  # 枠線の色を黒に設定
                ticks="inside",  # 目盛り線を内側に表示
                mirror=True,
                ticklen=10,
                tickwidth=2,
            ),
            legend=dict(
                x=0.02,          # プロット内の横位置（0.0: 左端、1.0: 右端）
                y=0.98,          # プロット内の縦位置（0.0: 下端、1.0: 上端）
                bgcolor="rgba(255, 255, 255, 0.7)",  # 背景色（透明度を調整）
                font=dict(size=cfg.fontsize_legend)  # フォントサイズを設定
            ),
            title="Least Squares Quadratic Fit",
            font=dict(family=cfg.font_graph, size=cfg.fontsize),
        )
    fig.show()

def plot(plot_mode, xlabels, ylabels, xdata_list, ydata_list, xlabels_cal, ylabels_cal, xcal_list, ycal_list):
    if plot_mode == "plotly":
        return plot_by_plotly(xlabels, ylabels, xdata_list, ydata_list, xlabels_cal, ylabels_cal, xcal_list, ycal_list):
    else:
        return plot_by_matplotlib(xlabels, ylabels, xdata_list, ydata_list, xlabels_cal, ylabels_cal, xcal_list, ycal_list):

def execute(cfg):
    """ 設定を基に計算を実行 """
    labels, data_list = read_data(cfg.infile)
    xlabels, ylabels, xdata_list, ydata_list = split_data(labels, data_list)
    xlabels_cal, ylabels_cal, xcal_list, ycal_list = analyze(xlabels, ylabels, xdata_list, ydata_list)
    save_data(xlabels, ylabels, xdata_list, ydata_list, xlabels_cal, ylabels_cal, xcal_list, ycal_list)
    plot(cfg.plot_mode, xlabels, ylabels, xdata_list, ydata_list, xlabels_cal, ylabels_cal, xcal_list, ycal_list)

def main():
    cfg, parser = initialize()
    cfg = update_vars(cfg, parser)
    if cfg.mode == 'exec':
        execute(cfg)
    else:
        print(f"\nError: Invalid mode [{cfg.mode}].\n")
        usage(parser)  # プログラム終了時に `argparse` の情報を動的に表示
        sys.exit(1)

    usage(parser)  # プログラム終了時に `argparse` の情報を動的に表示

if __name__ == "__main__":
    main()