#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
nonlinear_peak_fit.py

非線形最小二乗法で擬Voigt関数を使ってピークフィットを行うスクリプト。
- x, y データを Excel ファイルから読み込む
- コマンドライン引数で与えた（またはデフォルトの）初期ピークパラメータから「初期プロファイル」を計算
- scipy.optimize.curve_fit で最適化を実行
- 最適化パラメータから「最適化プロファイル」を計算
- 入力データ、初期プロファイル、最適化プロファイルをグラフ化
- フィッティング結果（パラメータ＋各波形）を Excel ファイルに保存
"""

import os
import sys
import argparse
from types import SimpleNamespace

import numpy as np
import pandas as pd
import matplotlib.pyplot as plt
import plotly.graph_objects as go
from scipy.optimize import curve_fit


def initialize():
    """ cfgオブジェクト作成およびコマンドライン引数パーサーの設定 """
    cfg = SimpleNamespace()

    parser = argparse.ArgumentParser(description="Pseudo-Voigt peak fitting using nonlinear least squares")
    parser.add_argument(
        "--infile", type=str, required=True,
        help="入力データ（Excel）のファイルパス。X列とY列がそれぞれ1列目・2列目にある想定。"
    )
    parser.add_argument(
        "--amp0", type=float, default=1.0,
        help="初期振幅 A0（デフォルト: 1.0）"
    )
    parser.add_argument(
        "--cen0", type=float, default=0.0,
        help="初期ピーク位置 x0（デフォルト: 0.0）"
    )
    parser.add_argument(
        "--fwhm0", type=float, default=1.0,
        help="初期ピーク幅（FWHM）（デフォルト: 1.0）"
    )
    parser.add_argument(
        "--eta0", type=float, default=0.5,
        help="初期混合比率 η（0～1）（デフォルト: 0.5）"
    )
    parser.add_argument(
        "--plot_mode", type=str, default="matplotlib",
        choices=["matplotlib", "plotly"], help="グラフ表示モード（デフォルト: matplotlib）"
    )
    parser.add_argument(
        "--fontsize", type=int, default=16,
        help="グラフのフォントサイズ（デフォルト: 16）"
    )
    parser.add_argument(
        "--fontsize_legend", type=int, default=12,
        help="グラフの凡例フォントサイズ（デフォルト: 12）"
    )
    parser.add_argument(
        "--font_graph", type=str, default="MS Gothic",
        help="グラフのフォント名（デフォルト: MS Gothic）"
    )
    parser.add_argument(
        "--outfile", type=str, default=None,
        help="フィッティング結果を保存する Excel ファイル（デフォルト: 入力ファイル名をベースに自動生成）"
    )

    return cfg, parser


def usage(parser):
    """ `argparse` を使って動的に使用方法を表示する """
    print("\n=== 使用方法 ===")
    parser.print_help()


def update_vars(cfg, parser):
    """ コマンドライン引数をパースして cfg にコピー """
    try:
        args = parser.parse_args()
        for key, value in vars(args).items():
            setattr(cfg, key, value)
    except argparse.ArgumentError:
        usage(parser)
        sys.exit(1)

    # outfile が指定されていない場合、infile のベース名から生成
    if not getattr(cfg, "outfile", None):
        infile_basename = os.path.splitext(cfg.infile)[0]
        cfg.outfile = f"{infile_basename}_fitting_results.xlsx"

    return cfg


def read_data(filename):
    """
    Excelファイルからデータを読み込む
    - filename: 読み込むExcelファイルのパス（1列目：x、2列目：yを想定）
    - return: 列ラベル (labels) とデータの NumPy 配列 (data_array)
    """
    try:
        df = pd.read_excel(filename)
        labels = df.columns.tolist()
        data_array = df.to_numpy()
        return labels, data_array
    except Exception as e:
        print(f"Error: データの読み込みに失敗しました: {e}")
        sys.exit(1)


def split_data(labels, data_array):
    """
    データを x, y に分割
    - labels: 列ラベルのリスト（例: ["x", "y"]）
    - data_array: 2列以上あると想定して、1列目を x、2列目を y に取り出す
    - return: xlabels, ylabels, xdata (1D numpy), ydata (1D numpy)
    """
    xlabels = labels[0]
    ylabels = labels[1]
    xdata = data_array[:, 0].astype(float)
    ydata = data_array[:, 1].astype(float)
    return xlabels, ylabels, xdata, ydata


def pseudo_voigt(x, A, x0, fwhm, eta):
    """
    擬Voigt関数（normalized form）
      - A    : 振幅
      - x0   : 中心位置
      - fwhm : full-width at half-maximum
      - eta  : mixing parameter (0≤eta≤1)
    正規化された形を用いる場合の式：
      G(x) = A * exp( -4 ln(2) * ((x - x0)/fwhm)^2 )
      L(x) = A * [1 / (1 + 4 * ((x - x0)/fwhm)^2)]
      PV(x) = eta * L(x) + (1 - eta) * G(x)
    """
    # ガウス成分
    G = A * np.exp(-4 * np.log(2) * ((x - x0) / fwhm) ** 2)
    # ローレンツ成分
    L = A * (1.0 / (1.0 + 4.0 * ((x - x0) / fwhm) ** 2))
    return eta * L + (1.0 - eta) * G


def compute_initial_profile(xdata, cfg):
    """
    初期ピークパラメータを用いて「初期プロファイル」を計算
    - xdata : 1D numpy array（観測点）
    - cfg    : cfg.amp0, cfg.cen0, cfg.fwhm0, cfg.eta0 が設定済み
    - return: y_init (1D numpy array)
    """
    A0 = cfg.amp0
    x0 = cfg.cen0
    fwhm0 = cfg.fwhm0
    eta0 = cfg.eta0
    y_init = pseudo_voigt(xdata, A0, x0, fwhm0, eta0)
    return y_init


def fit_peak(xdata, ydata, cfg):
    """
    curve_fit を用いて最適化を行う
    - xdata, ydata: 1D numpy arrays
    - cfg         : cfg.amp0, cfg.cen0, cfg.fwhm0, cfg.eta0 が設定済み
    - return: popt (最適化パラメータ), perr (パラメータの推定誤差)
    """
    # 初期パラメータベクトル
    p0 = [cfg.amp0, cfg.cen0, cfg.fwhm0, cfg.eta0]

    # 非線形最小二乗フィッティング
    try:
        popt, pcov = curve_fit(
            pseudo_voigt,
            xdata,
            ydata,
            p0=p0,
            bounds=(
                [0, np.min(xdata), 0, 0],      # A>=0, x0>=min(x), fwhm>=0, eta>=0
                [np.inf, np.max(xdata), np.inf, 1]  # eta<=1
            )
        )
        perr = np.sqrt(np.diag(pcov))
    except Exception as e:
        print(f"Error: フィッティングに失敗しました: {e}")
        sys.exit(1)

    return popt, perr


def save_data(cfg, xdata, ydata, y_init, y_opt, popt, perr):
    """
    フィッティング結果を Excel に保存
    - cfg     : cfg.outfile に保存先パスが設定済み
    - xdata   : 1D numpy array（入力 x データ）
    - ydata   : 1D numpy array（入力 y データ）
    - y_init  : 1D numpy array（初期プロファイル）
    - y_opt   : 1D numpy array（最適化プロファイル）
    - popt    : フィッティング済みパラメータ（A, x0, fwhm, eta）
    - perr    : パラメータの標準誤差
    """
    try:
        # 1) データシート：x, y, y_init, y_opt
        df_waveforms = pd.DataFrame({
            "x": xdata,
            "y_observed": ydata,
            "y_initial": y_init,
            "y_fitted": y_opt
        })

        # 2) パラメータシート：フィッティングパラメータと誤差
        param_names = ["A", "x0", "fwhm", "eta"]
        df_params = pd.DataFrame({
            "parameter": param_names,
            "value": popt,
            "stderr": perr
        })

        # ExcelWriter でまとめて保存
        with pd.ExcelWriter(cfg.outfile, engine="openpyxl") as writer:
            df_waveforms.to_excel(writer, sheet_name="waveforms", index=False)
            df_params.to_excel(writer, sheet_name="fit_parameters", index=False)

        print(f"フィッティング結果を '{cfg.outfile}' に保存しました。")
    except Exception as e:
        print(f"Error: 結果の保存に失敗しました: {e}")
        sys.exit(1)


def plot_by_matplotlib(cfg, xlabels, ylabels, xdata, ydata, y_init, y_opt):
    """
    matplotlib を使って入力データ・初期プロファイル・最適化プロファイルをプロット
    """
    plt.rcParams["font.family"] = cfg.font_graph
    plt.figure(figsize=(8, 6))

    # 元データは散布図
    plt.scatter(xdata, ydata, label="Original Data", color="blue", s=20)

    # 初期プロファイル（破線）
    plt.plot(xdata, y_init, color="orange", linestyle="--", linewidth=2, label="Initial Profile")

    # 最適化プロファイル（実線）
    plt.plot(xdata, y_opt, color="red", linestyle="-", linewidth=2, label="Fitted Profile")

    plt.xlabel(xlabels, fontsize=cfg.fontsize)
    plt.ylabel(ylabels, fontsize=cfg.fontsize)
    plt.title("Pseudo-Voigt Peak Fitting", fontsize=cfg.fontsize)
    plt.xticks(fontsize=cfg.fontsize)
    plt.yticks(fontsize=cfg.fontsize)
    plt.legend(fontsize=cfg.fontsize_legend)
    plt.grid(alpha=0.3)
    plt.tight_layout()
    plt.show()


def plot_by_plotly(cfg, xlabels, ylabels, xdata, ydata, y_init, y_opt):
    """
    plotly を使って入力データ・初期プロファイル・最適化プロファイルをプロット
    """
    fig = go.Figure()

    # 元データ
    fig.add_trace(go.Scatter(
        x=xdata,
        y=ydata,
        mode="markers",
        name="Original Data",
        marker=dict(color="blue", size=6),
    ))

    # 初期プロファイル
    fig.add_trace(go.Scatter(
        x=xdata,
        y=y_init,
        mode="lines",
        name="Initial Profile",
        line=dict(color="orange", dash="dash", width=2),
    ))

    # 最適化プロファイル
    fig.add_trace(go.Scatter(
        x=xdata,
        y=y_opt,
        mode="lines",
        name="Fitted Profile",
        line=dict(color="red", width=2),
    ))

    fig.update_layout(
        title="Pseudo-Voigt Peak Fitting",
        xaxis=dict(
            title=xlabels,
            titlefont=dict(size=cfg.fontsize),
            tickfont=dict(size=cfg.fontsize),
            showgrid=True,
            zeroline=False
        ),
        yaxis=dict(
            title=ylabels,
            titlefont=dict(size=cfg.fontsize),
            tickfont=dict(size=cfg.fontsize),
            showgrid=True,
            zeroline=False
        ),
        legend=dict(font=dict(size=cfg.fontsize_legend)),
        font=dict(family=cfg.font_graph, size=cfg.fontsize),
        width=800,
        height=600,
        plot_bgcolor="white"
    )
    fig.show()


def plot(cfg, xlabels, ylabels, xdata, ydata, y_init, y_opt):
    """
    プロット呼び出し用関数。plot_mode に応じて matplotlib か plotly を選択。
    """
    if cfg.plot_mode == "plotly":
        plot_by_plotly(cfg, xlabels, ylabels, xdata, ydata, y_init, y_opt)
    else:
        plot_by_matplotlib(cfg, xlabels, ylabels, xdata, ydata, y_init, y_opt)


def execute(cfg):
    """ 設定をもとにフィッティングを実行 → 保存 → プロット """
    # 1) データ読み込み
    labels, data_array = read_data(cfg.infile)
    xlabels, ylabels, xdata, ydata = split_data(labels, data_array)

    # 2) 初期プロファイル計算
    y_init = compute_initial_profile(xdata, cfg)

    # 3) フィッティング実行
    popt, perr = fit_peak(xdata, ydata, cfg)

    # 4) フィッティング後のプロファイル計算
    y_opt = pseudo_voigt(xdata, *popt)

    # 5) 結果保存
    save_data(cfg, xdata, ydata, y_init, y_opt, popt, perr)

    # 6) プロット
    plot(cfg, xlabels, ylabels, xdata, ydata, y_init, y_opt)


def main():
    cfg, parser = initialize()
    cfg = update_vars(cfg, parser)

    execute(cfg)


if __name__ == "__main__":
    main()
