#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
擬 Voigt 関数によるピークフィッティング（非線形最小二乗法）

修正版ポイント:
  - load_data_from_excel() が sheet_name=None のときも辞書を返さないように調整
  - ファイル拡張子が .csv なら pd.read_csv() を自動で呼ぶようにしてみた
  - エラー時のメッセージを詳しくして、問題個所をわかりやすくした
"""

import os
import numpy as np
import pandas as pd
import matplotlib.pyplot as plt
from scipy.optimize import curve_fit


#---------------------------------
# 1) 擬 Voigt 関数の定義
#---------------------------------
def pseudo_voigt(x, height, center, fwhm, eta):
    """
    擬 Voigt (Pseudo-Voigt) 関数
      G(x) = exp(-4 ln(2) * ((x - center)^2) / fwhm^2)
      L(x) = 1 / (1 + 4 * ((x - center)^2) / fwhm^2)
      f(x) = height * [ eta * L(x) + (1 - eta) * G(x) ]
    """
    gauss = np.exp(-4.0 * np.log(2) * ((x - center) ** 2) / fwhm**2)
    lorentz = 1.0 / (1.0 + 4.0 * ((x - center) ** 2) / fwhm**2)
    return height * (eta * lorentz + (1.0 - eta) * gauss)


#---------------------------------
# 2) データ読み込み関数の修正
#---------------------------------
def load_data_from_excel_or_csv(filename, sheet_name=None, x_col="x", y_col="y"):
    """
    ・拡張子が .xlsx/.xls → pandas.read_excel
    ・拡張子が .csv      → pandas.read_csv
    
    sheet_name=None のときは「最初のシートだけ」を読み込む
    （pd.read_excel ファイル全体を辞書で返したくないためのガード）
    """
    ext = os.path.splitext(filename)[1].lower()
    
    # --- Excel (.xlsx, .xls) の場合 ---
    if ext in [".xls", ".xlsx"]:
        try:
            if sheet_name is None:
                # 最初のシートだけ読み込み (sheet_name=0)
                df = pd.read_excel(filename, sheet_name=0)
            else:
                df = pd.read_excel(filename, sheet_name=sheet_name)
        except Exception as e:
            raise RuntimeError(f"Excel ファイルの読み込み中にエラーが発生しました: {e}")
    
    # --- CSV (.csv) の場合 ---
    elif ext == ".csv":
        try:
            df = pd.read_csv(filename)
        except Exception as e:
            raise RuntimeError(f"CSV ファイルの読み込み中にエラーが発生しました: {e}")
    
    else:
        raise RuntimeError(f"サポートしていない拡張子です: {ext}  (.xlsx, .xls, .csv のいずれかを指定してください)")
    
    # 列名チェック
    if x_col not in df.columns or y_col not in df.columns:
        raise KeyError(f"指定列 '{x_col}' または '{y_col}' がファイル内に見つかりません。実際の列名を確認してください。")
    
    x = df[x_col].to_numpy(dtype=float)
    y = df[y_col].to_numpy(dtype=float)
    return x, y


#---------------------------------
# 3) メイン処理
#---------------------------------
def main():
    # --- 3-1) ユーザーが編集すべき部分 ---
    # 読み込むファイル名 (Excel または CSV) と列名だけ変更してください
    data_file = "peak.xlsx"       # あるいは "data.csv"
    sheet_name = None             # Excel の場合に特定シートがあれば "Sheet1" などを指定。不要なら None
    x_col = "x"
    y_col = "y"
    
    # 初期パラメータ (適宜調整)
    initial_height = 1.0
    initial_center = 0.0
    initial_fwhm = 1.0
    initial_eta = 0.5
    p0 = [initial_height, initial_center, initial_fwhm, initial_eta]
    
    # パラメータ制約
    lower_bounds = [0.0,      -np.inf,    0.0,      0.0]
    upper_bounds = [np.inf,    np.inf,    np.inf,   1.0]
    bounds = (lower_bounds, upper_bounds)
    
    # --- 3-2) データ読み込み ---
    try:
        xdata, ydata = load_data_from_excel_or_csv(data_file, sheet_name, x_col, y_col)
    except Exception as e:
        print(f"データ読み込みエラー: {e}")
        return
    
    # --- 3-3) 初期プロファイル計算 ---
    y_init = pseudo_voigt(xdata, *p0)
    
    # --- 3-4) 非線形最小二乗法フィッティング ---
    try:
        popt, pcov = curve_fit(
            pseudo_voigt,
            xdata,
            ydata,
            p0=p0,
            bounds=bounds,
            maxfev=10000
        )
    except RuntimeError as err:
        print(f"フィッティングが収束しませんでした: {err}")
        return
    
    height_fit, center_fit, fwhm_fit, eta_fit = popt
    print("=== フィッティング結果 ===")
    print(f"  height = {height_fit:.6f}")
    print(f"  center = {center_fit:.6f}")
    print(f"   fwhm  = {fwhm_fit:.6f}")
    print(f"    eta  = {eta_fit:.6f}")
    print("===========================")
    
    # --- 3-5) 最適化プロファイル計算 ---
    y_fit = pseudo_voigt(xdata, *popt)
    
    # --- 3-6) プロット ---
    plt.figure(figsize=(8, 5))
    plt.scatter(xdata, ydata, s=20, color="black", label="Data", zorder=3)
    plt.plot(xdata, y_init, "--", color="gray", linewidth=1.5, label="Initial Guess")
    plt.plot(xdata, y_fit, "-", color="red", linewidth=2.0, label="Fitted Curve")
    
    plt.xlabel("x")
    plt.ylabel("y")
    plt.title("Pseudo-Voigt Peak Fit")
    plt.legend()
    plt.tight_layout()
    plt.show()


if __name__ == "__main__":
    main()
