import numpy as np
import matplotlib.pyplot as plt

# 乱数シード（再現性のため）
np.random.seed(0)

# ------------------------------------------------------------
# 1. 人工データの生成
# ------------------------------------------------------------
N = 50  # サンプル数

# 真のパラメータ (beta0: 切片, beta1, beta2)
beta_true = np.array([2.0, 1.5, -0.5])

# 説明変数 x1, x2 を等間隔に生成
x1 = np.linspace(0, 10, N)
x2 = x1**2

# 説明変数行列 X: (N×3) [1, x1, x2]
X = np.column_stack((np.ones(N), x1, x2))

# ノイズを含む目的変数 y
noise = np.random.normal(loc=0.0, scale=2.0, size=N)
y = beta_true[0] + beta_true[1]*x1 + beta_true[2]*x2 + noise

# ------------------------------------------------------------
# 2. 最小二乗法でパラメータとその誤差を推定
# ------------------------------------------------------------
XtX = X.T @ X
XtX_inv = np.linalg.inv(XtX)
beta_est = XtX_inv @ (X.T @ y)

residuals = y - X @ beta_est
p = X.shape[1]
RSS = residuals.T @ residuals
sigma2 = RSS / (N - p)
cov_beta = sigma2 * XtX_inv
beta_std = np.sqrt(np.diag(cov_beta))

# ------------------------------------------------------------
# 3. yの分布を解析的に求める（誤差伝播公式）
# ------------------------------------------------------------
y_mean = X @ beta_est                    # 各xにおけるyの期待値
y_var  = np.sum(X @ cov_beta * X, axis=1) # 各xにおけるyの分散
y_std  = np.sqrt(y_var)                  # 各xにおけるyの標準偏差

# ------------------------------------------------------------
# 4. 結果の表示（コンソール）
# ------------------------------------------------------------
print("True parameters:     ", beta_true)
print("Estimated parameters:", beta_est)
print("Std error of params: ", beta_std)

# ------------------------------------------------------------
# 5. 結果の表示（x1順にソートして描画）
# ------------------------------------------------------------
# ソート用インデックス
idx = np.argsort(x1)
x1_s = x1[idx]
y_s  = y[idx]
y_mean_s = y_mean[idx]
y_std_s  = y_std[idx]
line_true = beta_true[0] + beta_true[1]*x1_s + beta_true[2]*x2[idx]
line_est  = beta_est[0]  + beta_est[1]*x1_s  + beta_est[2]*x2[idx]

plt.figure(figsize=(10, 6))
plt.scatter(x1_s, y_s, color='blue', label='Data')
plt.plot   (x1_s, line_true, color='green', label='True regression line')
plt.plot   (x1_s, line_est,  color='red',   label='Estimated regression line')
plt.fill_between(x1_s, y_mean_s - y_std_s, y_mean_s + y_std_s,
                 color='gray', alpha=0.3, label='1σ predictive range')
plt.title('Distribution of y based on Parameter Uncertainty (Analytical)')
plt.xlabel('x1')
plt.ylabel('y')
plt.legend()
plt.show()
