import numpy as np
import matplotlib.pyplot as plt

# 乱数シード（再現性のため）
np.random.seed(0)

# ------------------------------------------------------------
# 1. 人工データの生成
# ------------------------------------------------------------
N = 50  # サンプル数
beta_true = np.array([2.0, 1.5, -0.5])  # 真のパラメータ

x1 = np.linspace(0, 10, N)
x2 = x1**2
X = np.column_stack((np.ones(N), x1, x2))

noise = np.random.normal(0, 2.0, size=N)
y = beta_true[0] + beta_true[1]*x1 + beta_true[2]*x2 + noise

# ------------------------------------------------------------
# 2. 最小二乗法でパラメータ推定と共分散行列の計算
# ------------------------------------------------------------
XtX    = X.T @ X
XtX_inv= np.linalg.inv(XtX)
beta_est = XtX_inv @ (X.T @ y)

residuals = y - X @ beta_est
p = X.shape[1]
sigma2 = (residuals @ residuals) / (N - p)
cov_beta= sigma2 * XtX_inv
beta_std = np.sqrt(np.diag(cov_beta))

# ------------------------------------------------------------
# 3. パラメータをサンプリングして回帰曲線をプロット
# ------------------------------------------------------------
num_samples = 5000
beta_samples = np.random.multivariate_normal(beta_est, cov_beta, num_samples)

# ソート用インデックス
idx   = np.argsort(x1)
x1_s  = x1[idx]
x2_s  = x2[idx]

plt.figure(figsize=(10,6))
plt.scatter(x1, y, color='blue', label='Data')

# ±5σ 内の曲線（薄いグレー）
for b in beta_samples:
    if np.all(np.abs(b - beta_est) <= beta_std * 5):
        y_pred = b[0] + b[1]*x1_s + b[2]*x2_s
        plt.plot(x1_s, y_pred, color='yellow', alpha=0.02)

# ±1σ 内の曲線（やや濃いグレー）
for b in beta_samples:
    if np.all(np.abs(b - beta_est) <= beta_std * 1):
        y_pred = b[0] + b[1]*x1_s + b[2]*x2_s
        plt.plot(x1_s, y_pred, color='cyan', alpha=0.02)

# 推定回帰直線
y_est = beta_est[0] + beta_est[1]*x1_s + beta_est[2]*x2_s
plt.plot(x1_s, y_est, color='red', linewidth=2, label='Estimated regression line')

# 真の回帰直線
y_true = beta_true[0] + beta_true[1]*x1_s + beta_true[2]*x2_s
plt.plot(x1_s, y_true, color='green', linestyle='--', label='True regression line')

plt.title('Regression Curves: ±1σ (cyan) & ±5σ (yellow)')
plt.xlabel('x1')
plt.ylabel('y')
plt.legend()
plt.show()

# ------------------------------------------------------------
# 4. コンソール出力
# ------------------------------------------------------------
print("True parameters:     ", beta_true)
print("Estimated parameters:", beta_est)
print("Std error of params: ", beta_std)
