"""
速度分布に従うMetropolis-HastingsサンプラーとCLT・分布比較可視化プログラム

本プログラムは、速度分布の例として
    f(x) = x^2 * exp(-a * x^2)   (x ≥ 0)
を対象とします。ここではパラメータ a = 1 とし、
f(x) = x^2 * exp(-x^2) として扱います。

正規化定数は
    Z = 4/√π
となるため、正規化済みの理論分布は

    f_norm(x) = (4/√π) * x^2 * exp(-x^2)

となります。

本プログラムでは以下の2点を実施します：
1. Metropolis-Hastings法（反射付き提案）により f(x) に従うサンプルを生成し、  
   サンプルのヒストグラム（density=True）と正規化済み理論分布 f_norm(x) を [0,5] の横軸で重ね合わせて表示します。

2. 複数回の試行（N_runs回）で得られた各実験のサンプル平均の分布を算出し、  
   サンプル数 n が大きくなるにつれて、サンプル平均の分布が正規分布 N(μ,σ/√n) に収束する（中心極限定理）様子を可視化します。

グラフ表示には "MS Gothic" フォントを使用し、全てのグラフの横軸表示範囲は [0,5] に固定しています。
"""

import numpy as np
import matplotlib.pyplot as plt
from scipy.stats import norm

# 乱数の再現性のためのシード設定
np.random.seed(0)

# --- フォント設定 ---
plt.rcParams['font.family'] = 'MS Gothic'
plt.rcParams['axes.unicode_minus'] = False

# --- パラメータ ---
a = 1.0  # 速度分布のパラメータ

# --- 目標関数 ---
# 非正規化の速度分布: f(x)=x^2 * exp(-a*x^2) (x>=0)
def f(x):
    return x**2 * np.exp(-a*x**2) if x >= 0 else 0.0

# 正規化済みの理論分布
def f_norm(x):
    return (4/np.sqrt(np.pi)) * x**2 * np.exp(-a*x**2) if x >= 0 else 0.0

# --- 提案関数 ---
# 対称なガウス（平均0, 標準偏差 sigma）で提案し、もし提案値が負なら反射させる
def propose(x, sigma=0.3):
    x_prop = x + np.random.normal(0, sigma)
    if x_prop < 0:
        x_prop = -x_prop  # 反射処理
    return x_prop

# --- Metropolis-Hastingsサンプラー ---
def metropolis_hastings(f, n_samples, x0=1.0, sigma=0.3):
    """
    Metropolis-Hastings法によるサンプル生成関数

    Parameters:
        f         : 対象とする非正規化確率密度関数（x ≥ 0 で x^2 * exp(-a*x^2) を返す）
        n_samples : 生成するサンプル数
        x0        : 初期状態（x0 ≥ 0）
        sigma     : 提案分布の標準偏差

    Returns:
        samples   : 生成されたサンプルの numpy 配列
    """
    samples = []
    x = x0
    for _ in range(n_samples):
        x_prop = propose(x, sigma)
        alpha = min(1, f(x_prop) / f(x))
        if np.random.rand() < alpha:
            x = x_prop
        samples.append(x)
    return np.array(samples)

# === 1. サンプリング分布と理論分布の重ね合わせ ===

# 十分な数のサンプルを取得
n_samples_total = 50000
samples = metropolis_hastings(f, n_samples_total, x0=1.0, sigma=0.3)

plt.figure(figsize=(10, 6))
# ヒストグラム（density=Trueにより確率密度として表示）
plt.hist(samples, bins=100, density=True, alpha=0.6, label='サンプル分布')

# 理論分布 f_norm(x) を計算して重ね合わせる（横軸を [0,5] に固定）
x_vals = np.linspace(0, 5, 300)
pdf_vals = np.array([f_norm(x) for x in x_vals])
plt.plot(x_vals, pdf_vals, 'r-', label='理論分布')

plt.xlim(0, 5)
plt.xlabel('x')
plt.ylabel('確率密度')
plt.title('サンプリング分布と正規化理論分布の比較')
plt.legend()
plt.grid(True)
plt.tight_layout()
plt.show()

# === 2. 中心極限定理 (CLT) の可視化 ===
# 各実験において n 個のサンプルから算出したサンプル平均の分布を表示

# 理論上、正規化済み f(x) の平均は (4/√π) とはならず、本来の f(x) の平均は
# 速度分布の場合、正規化した分布 f_norm(x) = (4/√π) x^2 exp(-x^2)
# の平均は E[x] = 2/√π ≈ 1.12838、分散は Var[x] = 1.5 - (4/π) ≈ 0.22676 となります。
true_mean = 2 / np.sqrt(np.pi)         # 約 1.12838
true_std = np.sqrt(1.5 - 4/np.pi)        # 約 0.476

N_runs = 1000  # 試行回数
n_list = [10, 50, 100, 500, 1000]  # 各試行ごとのサンプル数

plt.figure(figsize=(14, 8))
for i, n in enumerate(n_list, 1):
    means = []
    for _ in range(N_runs):
        samples = metropolis_hastings(f, n_samples=n, x0=1.0, sigma=0.3)
        means.append(np.mean(samples))
    means = np.array(means)
    
    # 横軸は [0,5] に固定
    x_vals_clt = np.linspace(0, 5, 300)
    # CLTにより、サンプル平均は N(true_mean, true_std/√n) に収束
    pdf_vals_clt = norm.pdf(x_vals_clt, loc=true_mean, scale=true_std/np.sqrt(n))
    
    plt.subplot(2, 3, i)
    plt.hist(means, bins=30, density=True, alpha=0.6, label=f'n = {n}')
    plt.plot(x_vals_clt, pdf_vals_clt, 'r-', label='正規分布')
    plt.xlim(0, 5)
    plt.xlabel('サンプル平均')
    plt.ylabel('密度')
    plt.title(f'n = {n}, 標準偏差 ≈ {np.std(means):.3f}')
    plt.legend()
    plt.grid(True)

plt.suptitle('速度分布 f(x)=x^2 exp(-a*x^2) に従うMHサンプラーによる\n中心極限定理の可視化', fontsize=16)
plt.tight_layout(rect=[0, 0, 1, 0.95])
plt.show()
