import numpy as np
import matplotlib.pyplot as plt

def erf_inv(z):
    """手作り逆誤差関数（Winitzkiの近似式）"""
    a = 0.147
    sign = np.sign(z)
    ln = np.log(1 - z**2)
    first = 2/(np.pi * a) + ln/2
    second = ln/a
    return sign * np.sqrt( np.sqrt(first**2 - second) - first )

def normal_ppf(u):
    """手作り正規分布の逆累積分布関数 (標準正規)"""
    z = 2*u - 1
    return np.sqrt(2) * erf_inv(z)

def generate_normal_samples(n_samples, mu=0.0, sigma=1.0, eps=1e-10):
    """逆関数法＋手作り逆誤差関数で正規乱数生成"""
    u = np.random.uniform(0, 1, n_samples)
    u = np.clip(u, eps, 1 - eps)
    samples_standard = normal_ppf(u)  # 標準正規乱数
    samples = mu + sigma * samples_standard
    return samples

# パラメータ
n_samples = 10000
mu = 2.0      # 平均
sigma = 1.5   # 標準偏差

# サンプリング
samples = generate_normal_samples(n_samples, mu, sigma)

# 理論的な正規分布
from scipy.stats import norm
x = np.linspace(mu - 4*sigma, mu + 4*sigma, 1000)
fx = norm.pdf(x, loc=mu, scale=sigma)

# 可視化
plt.hist(samples, bins=50, density=True, alpha=0.6, label="Sampled")
plt.plot(x, fx, 'r-', label="Theoretical PDF", linewidth=2)
plt.xlabel('x')
plt.ylabel('Density')
plt.legend()
plt.title('Inverse Transform Sampling for Normal(μ=2, σ=1.5)')
plt.grid(True)
plt.show()
