import numpy as np
import matplotlib.pyplot as plt

def generate_exponential_samples_direct(n_samples, lam=1.0):
    """
    直接変換法 (Direct Transformation Method) による指数分布乱数生成
    lam : 指数分布のλ (平均=1/λ)
    """
    y = np.random.uniform(0, 1/lam, n_samples)  # f(x) = λ exp(-λx) の最大値1/λの範囲で一様乱数
    x = -np.log(1 - lam * y) / lam  # 逆変換
    return x

# パラメータ
n_samples = 10000
lam = 2.0  # λ=2の指数分布

# サンプル生成
samples = generate_exponential_samples_direct(n_samples, lam)

# 理論分布
x = np.linspace(0, 3, 1000)
fx = lam * np.exp(-lam * x)

# 可視化
plt.hist(samples, bins=50, density=True, alpha=0.6, label="Sampled (Direct Transformation)")
plt.plot(x, fx, 'r-', label="Theoretical PDF", linewidth=2)
plt.xlabel('x')
plt.ylabel('Density')
plt.legend()
plt.title('Exponential Distribution Sampling (Direct Transformation)')
plt.grid(True)
plt.show()
