import numpy as np
import matplotlib.pyplot as plt
from scipy.stats import norm

# フォント設定
fontsize = 16
plt.rcParams['font.family'] = 'MS Gothic'
plt.rcParams['font.size'] = fontsize


# パラメータの設定
mu = 0  # 平均
sigma = 1  # 標準偏差
n = 1  # 最大σ範囲

# xの値を生成
x = np.linspace(mu - 4*sigma, mu + 4*sigma, 500)
y = norm.pdf(x, mu, sigma)

# グラフの準備
fig, axes = plt.subplots(1, 2, figsize=(12, 6))
#axes[1].grid(visible=False)  # グリッド線を完全非表示に
#axes[1].tick_params(grid_color="none")  # グリッド線に影響しない設定
#axes[1].tick_params(axis='both', which='major', length=5)

# 左のグラフ: 正規分布と信頼区間
x_fill = np.linspace(mu - n*sigma, mu + n*sigma, 500)
y_fill = norm.pdf(x_fill, mu, sigma)
axes[0].plot(x, y, color="blue")
axes[0].fill_between(x_fill, 0, y_fill, color="orange", alpha=0.5, label=f"{n}σ 信頼区間")
axes[0].text(mu - n*sigma, 0.02, "区間外", horizontalalignment="right", color="red", fontsize=12)
axes[0].text(mu + n*sigma, 0.02, "区間外", horizontalalignment="left",  color="red", fontsize=12)
#axes[0].set_title("正規分布と信頼区間")
axes[0].set_xlabel("値")
axes[0].set_ylabel("確率密度")
axes[0].legend(loc="upper right")
#axes[0].grid(True)

# 右のグラフ: σの係数 vs 区間外確率
sigma_values = np.linspace(0.1, 4, 500)  # 0.1から始めてlog軸に対応
outside_prob = 2 * (1 - norm.cdf(sigma_values))  # 両側の外れ確率
axes[1].plot(sigma_values, outside_prob, label="区間外確率", color="green")
axes[1].set_xlim([0, 4])
axes[1].set_yscale("log")  # y軸を対数スケールに設定

# 右のグラフ: 信頼区間の線とラベル
sigma_points = [1, 2, 3]
for sigma in sigma_points:
    prob_outside = 2 * (1 - norm.cdf(sigma))
#    axes[1].axvline(sigma, color="red", linestyle="--", alpha=0.7, label=f"{sigma}σ 信頼区間外")
    axes[1].hlines(prob_outside, xmin=0, xmax=sigma, colors="blue", linestyles="--", alpha=0.7)
    axes[1].text(
        0,  # 水平線の左端
        prob_outside,  # y軸の位置を prob_outside に設定
        f"{prob_outside*100:.2f}% ({sigma:.1f}$\sigma$)",  # 表示するラベル
        color="blue",
        fontsize=fontsize,
        verticalalignment="top"  # 上端を基準に配置
    )

# 信頼区間の線とラベル
confidence_levels = [0.9, 0.95, 0.99]  # 信頼区間
sigma_lines = [norm.ppf((1 + cl) / 2) for cl in confidence_levels]  # 対応するσ
for i, (sigma_line, cl) in enumerate(zip(sigma_lines, confidence_levels)):
    axes[1].axvline(sigma_line, color="red", linestyle="--", alpha=0.7)
    axes[1].text(
        sigma_line, 
        1e-4, 
        f"{int(cl*100)}% ({sigma_line:.3f}$\sigma$)", 
        color="red", 
        fontsize=fontsize, 
        rotation=90, 
        verticalalignment="bottom"
    )

# 装飾
#axes[1].set_title("σの係数 vs 区間外確率")
axes[1].set_xlabel("σの係数", fontsize = fontsize)
axes[1].set_ylabel("区間外確率", fontsize = fontsize)
axes[1].legend(fontsize = fontsize)
#axes[1].grid(True)

# 全体表示
plt.tight_layout()
plt.show()
