# bayesian_nonlinear_regression_mcmc.py

"""
非線形ベイズ回帰（MCMC）による事後推定と予測の可視化

モデル: y = a * exp(b * x) + ε,   ε ~ N(0, σ^2)
"""

import os
import numpy as np
import multiprocessing
import matplotlib.pyplot as plt
import pymc as pm
import arviz as az

# cpuのコア数を取得
try:
    ncores = multiprocessing.cpu_count()
except NotImplementedError:
    ncores = os.cpu_count()
if ncores is None:
    ncores = 1

# for debug（Windowsでは1にすべき）
ncores = 1

# matplotlibでMS Gothicを使用（日本語ラベルのため）
plt.rcParams['font.family'] = 'MS Gothic'
plt.rcParams['axes.unicode_minus'] = False  # マイナス記号の表示を有効にする


def main():
    # 1. データ生成
    np.random.seed(0)
    x = np.linspace(0, 1, 20)
    true_a, true_b = 1.0, 2.0
    sigma_noise = 0.2
    y = true_a * np.exp(true_b * x) + np.random.normal(0, sigma_noise, size=x.shape)

    # 2. PyMCモデルの定義
    with pm.Model() as model:
        a = pm.Normal("a", mu=0, sigma=10)
        b = pm.Normal("b", mu=0, sigma=10)
        sigma = pm.HalfNormal("sigma", sigma=1)
        mu = a * pm.math.exp(b * x)
        y_obs = pm.Normal("y_obs", mu=mu, sigma=sigma, observed=y)
        trace = pm.sample(cores=ncores, draws=2000, tune=1000, return_inferencedata=True, target_accept=0.9)

    # 3. パラメータの事後分布の可視化
    az.plot_trace(trace, var_names=["a", "b", "sigma"])
    plt.suptitle("パラメータの事後分布とマルコフ連鎖の推移", fontsize=14)
    plt.tight_layout()
    plt.show()
#    plt.pause(0.1)

    # 4. 事後平均による予測曲線
    a_mean = trace.posterior['a'].mean().item()
    b_mean = trace.posterior['b'].mean().item()

    x_plot = np.linspace(0, 1, 100)
    y_mean_pred = a_mean * np.exp(b_mean * x_plot)

    plt.figure(figsize=(8, 5))
    plt.scatter(x, y, label="観測データ", color="black")
    plt.plot(x_plot, y_mean_pred, label="事後平均による予測", color="red")
    plt.xlabel("x")
    plt.ylabel("y")
    plt.title("ベイズ回帰モデルの予測（事後平均）")
    plt.legend()
    plt.grid(True)
    plt.show()
#    plt.pause(0.1)

    # 5. 信頼区間（ノイズなしの予測）
    a_samples = trace.posterior['a'].stack(sample=("chain", "draw")).values
    b_samples = trace.posterior['b'].stack(sample=("chain", "draw")).values
    y_preds = np.array([a * np.exp(b * x_plot) for a, b in zip(a_samples, b_samples)])
    y_hpd = az.hdi(y_preds, hdi_prob=0.95)

    plt.figure(figsize=(8, 5))
    plt.scatter(x, y, label="観測データ", color="black")
    plt.plot(x_plot, y_mean_pred, label="事後平均", color="red")
    plt.fill_between(x_plot, y_hpd[:, 0], y_hpd[:, 1], color="red", alpha=0.3, label="95% 信頼区間")
    plt.xlabel("x")
    plt.ylabel("y")
    plt.title("事後予測と不確かさの可視化（ノイズなし）")
    plt.legend()
    plt.grid(True)
    plt.show()
#    plt.pause(0.1)

# 6. ノイズ込みの予測（posterior predictive）
    with model:
        posterior_predictive = pm.sample_posterior_predictive(trace, var_names=["y_obs"], random_seed=0)

# (chain, draw, observed_dim) → (samples, x数)
    y_pp_samples = posterior_predictive.posterior_predictive["y_obs"].values
    n_samples, n_points = y_pp_samples.shape[0] * y_pp_samples.shape[1], y_pp_samples.shape[2]
    y_pp_samples_2d = y_pp_samples.reshape(n_samples, n_points)  # shape = (samples, x数)

# 95% 高密度区間（ノイズあり）
    y_pp_hdi = az.hdi(y_pp_samples_2d, hdi_prob=0.95)

# 可視化
    plt.figure(figsize=(8, 5))
    plt.scatter(x, y, label="観測データ", color="black")
    plt.fill_between(x, y_pp_hdi[:, 0], y_pp_hdi[:, 1], color="blue", alpha=0.3, label="95% 予測信頼区間（ノイズ込み）")
    plt.plot(x, y_pp_samples_2d.mean(axis=0), color="blue", label="予測平均（ノイズ込み）")
    plt.xlabel("x")
    plt.ylabel("y")
    plt.title("事後予測分布の信頼区間（ノイズ込み）")
    plt.legend()
    plt.grid(True)
    plt.pause(0.1)

# Windows向けに必須のガード
if __name__ == '__main__':
    main()
    input("Press Enter to exit...")
